﻿// <copyright file="DockPanelController.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>
// 


/*
* @class DockPanelController controller class for DockPanel control
*/
define(['common/Enums', 'base/ContainerController'], function (Enums, ContainerController) {

    var DockPanelController = ContainerController.extend({

        init: function () {
            this._super();

            this.ClassName = 'DockPanelController';
        },

        onAddedToDOM: function () {
            this._super();
            this.fullReposition();
        },

        fullReposition: function () {
            this.repositionControls(0);
        },

        onChildrenAdded: function (event) {

            var controlsCount = this.orderedControls.length;
            var startIndex = this.mvc.model.getLastChildFill() ? controlsCount - 1 : controlsCount;

            this._super(event);

            if (event.childModels && event.childModels.length > 0) {
                this.repositionControls(startIndex);
            }
        },

        modelPropertyChangedInternal: function (event) {

            this._super(event);

            var value = this.mvc.model.get(event.property);

            switch (event.property) {
                case Enums.ParameterRoles.LAST_CHILD_FILL:
                    this.onLastChildFillChanged(value);
                    break;
            }
        },

        onLastChildFillChanged: function (value) {

            if (this.orderedControls.length === 0) {
                return;
            }

            //adjust last control only
            this.repositionControls(this.orderedControls.length - 1);
        },

        onBorderThiknessChanged: function (value) {
            this._super();

            this.repositionControls(0);
        },

        childModelPropertyChangedInternal: function (event) {
            this._super();

            switch (event.property) {
                case Enums.ParameterRoles.DOCK_STYLE:
                    this.onChildDockChanged(event);
                    break;
                case Enums.ParameterRoles.WIDTH:
                case Enums.ParameterRoles.WIDTH_UNITS:
                    this.onChildWidthChanged(event);
                    break;
                case Enums.ParameterRoles.HEIGHT:
                case Enums.ParameterRoles.HEIGHT_UNITS:
                    this.onChildHeightChanged(event);
                    break;
                case Enums.ParameterRoles.IS_VISIBLE:
                    this.onChildVisibilityChanged(event);
                    break;
            }
        },

        childModelPropertyChanging: function (event) {
            switch (event.property) {
                case Enums.ParameterRoles.WIDTH:
                case Enums.ParameterRoles.WIDTH_UNITS:
                    this.onChildChangingWidth(event);
                    break;
                case Enums.ParameterRoles.HEIGHT:
                case Enums.ParameterRoles.HEIGHT_UNITS:
                    this.onChildChangingHeight(event);
                    break;
            }
        },

        onChildChangingWidth: function (event) {
            var childDock = this.getChildDockStyle(event.target);
            var childIndex = this.childIndexById(event.target.getId());

            if (childDock === Enums.DockType.Up || childDock === Enums.DockType.BOTTOM) {
                event.silent = true;
                return;
            }
            if (childIndex === this.orderedControls.length - 1 &&
                this.mvc.model.getLastChildFill()) {
                event.silent = true;
                return;
            }
        },

        onChildChangingHeight: function (event) {
            var childDock = this.getChildDockStyle(event.target);
            var childIndex = this.childIndexById(event.target.getId());

            if (childDock === Enums.DockType.Left || childDock === Enums.DockType.Right) {
                event.silent = true;
                return;
            }

            if (childIndex === this.orderedControls.length - 1 &&
                this.mvc.model.getLastChildFill()) {
                event.silent = true;
                return;
            }
        },      

        onChildDockChanged: function (event) {
            var childIndex = this.childIndexById(event.target.getId());

            if (childIndex === this.orderedControls.length - 1 && this.mvc.model.getLastChildFill()) {
                //ignore because this child should fill empty space
                return;
            }
            //TODO: call applyWidth for child control in case if it has changed orientation
            this.repositionControls(childIndex);
        },

        onChildVisibilityChanged: function (event) {
           
        },

        onChildWidthChanged: function (event) {
            var childDock = this.getChildDockStyle(event.target);
            var childIndex = this.childIndexById(event.target.getId());

            if (childDock === Enums.DockType.Up || childDock === Enums.DockType.Down) {
                //we should never come here because firing this event is blocked                 
                return;
            }

            if (childIndex === this.orderedControls.length - 1 &&
                this.mvc.model.getLastChildFill()) {
                //we should never come here because firing this event is blocked 
                return;
            }

            this.repositionControls(childIndex);
        },

        onChildHeightChanged: function (event) {
            var childDock = this.getChildDockStyle(event.target);
            var childIndex = this.childIndexById(event.target.getId());

            if (childDock === Enums.DockType.Left || childDock === Enums.DockType.Right) {
                //we should never come here because firing this event is blocked 
                return;
            }

            if (childIndex === this.orderedControls.length - 1 &&
                this.mvc.model.getLastChildFill()) {
                //we should never come here because firing this event is blocked 
                return;
            }

            this.repositionControls(childIndex);
        },

        fireActualWidthChanged: function (newValue) {
            this._super(newValue);
            this.repositionControls(0);
        },

        fireActualHeightChanged: function (newValue) {
            this._super(newValue);
            this.repositionControls(0);
        },

        repositionControls: function (startIndex) {

            if (startIndex < 0) {
                return;
            }

            if (this.orderedControls.length === 0) {
                return;
            }

            var i,
                childControl,
                childWidth,
                childHeight,
                emptyRect = this.calcEmptyRect(startIndex);

            if (emptyRect.width === 0 || emptyRect.width === 0) {
                //optimization
                return;
            }
            for (i = startIndex; i < this.orderedControls.length; i++) {
                childControl = this.orderedControls[i];
                if (this._skipControl(childControl)) {
                    continue;
                }
                if (i === this.orderedControls.length - 1 && this.mvc.model.getLastChildFill()) {
                    childControl.controller.setControlLeft(emptyRect.left + 'px');
                    childControl.controller.setControlTop(emptyRect.top + 'px');
                    childControl.controller.setControlWidth(emptyRect.width + 'px');
                    childControl.controller.setControlHeight(emptyRect.height + 'px');
                }
                else {
                    switch (this.getChildDockStyle(childControl.model)) {
                        case Enums.DockType.Left:
                            //abs because line may jave negative size
                            childWidth = Math.abs(childControl.controller.getControlWidthPx());
                            childControl.controller.setControlLeft(emptyRect.left + 'px');
                            childControl.controller.setControlTop(emptyRect.top + 'px');
                            childControl.controller.setControlHeight(emptyRect.height + 'px');
                            emptyRect.left += childWidth;
                            emptyRect.width -= childWidth;
                            break;
                        case Enums.DockType.Up:
                            childHeight = Math.abs(childControl.controller.getControlHeightPx());
                            childControl.controller.setControlLeft(emptyRect.left + 'px');
                            childControl.controller.setControlTop(emptyRect.top + 'px');
                            childControl.controller.setControlWidth(emptyRect.width + 'px');
                            emptyRect.top += childHeight;
                            emptyRect.height -= childHeight;
                            break;
                        case Enums.DockType.Right:
                            childWidth = Math.abs(childControl.controller.getControlWidthPx());
                            childControl.controller.setControlLeft(emptyRect.left + emptyRect.width - childWidth + 'px');
                            childControl.controller.setControlTop(emptyRect.top + 'px');
                            childControl.controller.setControlHeight(emptyRect.height + 'px');
                            emptyRect.width -= childWidth;
                            break;
                        case Enums.DockType.Down:
                            childHeight = Math.abs(childControl.controller.getControlHeightPx());
                            childControl.controller.setControlLeft(emptyRect.left + 'px');
                            childControl.controller.setControlTop(emptyRect.top + emptyRect.height - childHeight + 'px');
                            childControl.controller.setControlWidth(emptyRect.width + 'px');
                            emptyRect.height -= childHeight;
                            break;
                    }
                }
            }
        },

        _skipControl: function (childControl) {
            this._super(childControl);
        }, 

        getChildDockStyle: function(childControlModel)
        {
            return childControlModel.get(Enums.ParameterRoles.DOCK_STYLE);
        },     

        calcEmptyRect: function (startIndex) {
            var borderWidth = this.mvc.model.getBorderThickness(),
                left = 0,
                top = 0,
                width = this.getControlWidthPx() - borderWidth * 2,
                height = this.getControlHeightPx() - borderWidth * 2,
                i,
                control;

            for (i = 0; i < startIndex; i++) {
                control = this.orderedControls[i];

                switch (this.getChildDockStyle(control.model)) {
                    case Enums.DockType.Left:
                        left += control.controller.getControlWidthPx();
                        width -= left;
                        break;
                    case Enums.DockType.Up:
                        top += control.controller.getControlHeightPx();
                        height -= top;
                        break;
                    case Enums.DockType.Right:
                        width -= control.controller.getControlWidthPx();
                        break;
                    case Enums.DockType.Down:
                        height -= control.controller.getControlHeightPx();
                        break;
                }
            }

            return { left: left, top: top, width: width, height: height };
        }
    });

    return DockPanelController;
});
