﻿define(['common/Enums', 'base/ControlController', 'libs/svg/jquery_svg_drawing'], function (Enums, WindowController, svgApi) {

    var TruncatedConeController = WindowController.extend({

        init: function (statisticId) {
            this._super(statisticId);
            this.ClassName = 'TruncatedConeController';
            this.modelChanged[Enums.ParameterRoles.BACKGROUND_COLOR] = this.onFillColorChanged.bind(this);
            this.modelChanged[Enums.ParameterRoles.CONE_RATIO] = this.onResize.bind(this);
            this.modelChanged[Enums.ParameterRoles.CUT_THICKNESS] = this.onCutChanged.bind(this);
            this.modelChanged[Enums.ParameterRoles.CUT_COLOR] = this.onCutChanged.bind(this);
        },

        onAddedToDOM: function () {
            this._super();
            this.mvc.view.$().css('border', 'none');
            this.appendTruncatedCone();
        },

        appendTruncatedCone: function () {
            var controlHeight = this.mvc.model.getHeight();
            var controlWidth = this.mvc.model.getWidth();
            var clientId = this.clientId();
            this.mvc.view.$().attr('id', clientId);
            var svg = svgApi.getSVG(clientId);

            svg.configure({ viewBox: "0 0 " + controlWidth + " " + controlHeight }, false);
            this.svgRoot = svg;
            var sizes = this.getSizes();
            this.createTruncatedCone(svg, sizes);
            this.applyShadow();
        },

        getSizes: function () {
            var controlHeightPx = this.mvc.model.getHeight();
            var controlWidthPx = this.mvc.model.getWidth();
            var ratio = this.mvc.model.get(Enums.ParameterRoles.CONE_RATIO);
            if (ratio > 1) {
                topDiam = controlWidthPx;
                botDiam = topDiam / ratio;
            } else {
                botDiam = controlWidthPx;
                topDiam = botDiam * ratio;
            }

            var min = Math.min(topDiam, botDiam);

            var sizes = {
                min: min,
                max: controlWidthPx,
                topDiam: topDiam,
                botDiam: botDiam,
                height: controlHeightPx,
            };
            return sizes;
        },

        onFillColorChanged: function (event) {
            var svg = this.svgRoot;
            if (svg != null) {
                var colorMatrix = document.createElementNS('http://www.w3.org/2000/svg', 'feColorMatrix');
                colorMatrix.setAttribute('id', 'feColorMatrix')
                colorMatrix.setAttribute('values', event);
                var oldMatrix = svg._svg.getElementById('feColorMatrix');
                var parent = oldMatrix.parentNode;
                parent.replaceChild(colorMatrix, oldMatrix);
            }
        },

        onIsEnabledChanged: function () {
            this._super();
            this.applyShadow();
        },

        onCutChanged: function () {
            var svg = this.svgRoot;
            if (svg != null) {

                var fc = svg._svg.childNodes[0];
                while (fc) {
                    svg._svg.removeChild(fc);
                    fc = svg._svg.childNodes[0];
                }
                var sizes = this.getSizes();
                svg.configure({ viewBox: "0 0 " + sizes.max + " " + sizes.height }, false);
                this.createTruncatedCone(svg, sizes);
            }
        },

        applyShadow: function () {
            var shadow = {
                size: this.mvc.model.get(Enums.ParameterRoles.SHADOW_SIZE),
                color: this.mvc.model.getShadowColor()
            };
            var gray = this.mvc.model.getIsEnabled() ? 0 : 100;
            this.mvc.view.$().css('box-shadow', '');
            this.mvc.view.$().css('background-color', 'transparent');
            this.mvc.view.$().css('filter', String.format('drop-shadow({0}px {0}px {1}px {2}) grayscale({3}%)', shadow.size, 0, shadow.color, gray));
        },

        onResize: function () {
            var svg = this.svgRoot;
            if (svg != null) {

                var fc = svg._svg.childNodes[0];
                while (fc) {
                    svg._svg.removeChild(fc);
                    fc = svg._svg.childNodes[0];
                }
                var sizes = this.getSizes();
                svg.configure({ viewBox: "0 0 " + sizes.max + " " + sizes.height }, false);
                this.createTruncatedCone(svg, sizes);
            }
        },


        createTruncatedCone: function (svg, sizes) {
            var leftHalf = document.createElementNS('http://www.w3.org/2000/svg', 'path');
            var rightHalf = document.createElementNS('http://www.w3.org/2000/svg', 'path');

            var cutColor = this.mvc.model.get(Enums.ParameterRoles.CUT_COLOR);
            var cutThick = this.mvc.model.get(Enums.ParameterRoles.CUT_THICKNESS);

            var topDiam = Math.ceil(sizes.topDiam);
            var botDiam = Math.ceil(sizes.botDiam);

            var height = Math.ceil(sizes.height);
            var wide = Math.max(topDiam, botDiam);
            var narrow = Math.min(topDiam, botDiam);
            //five 'x' values
            var leftBot = (wide - botDiam) / 2;
            var leftTop = (wide - topDiam) / 2;
            var center = wide / 2
            var rightTop = wide - leftTop;
            var rightBot = wide - leftBot;
            var id = svg._container.id;
            var smooth = 1;

            var leftTrapPath = `M${leftTop + cutThick - smooth},0 L${leftBot + cutThick - smooth},${height} L${center + smooth},${height} L${center + smooth},0`;
            leftHalf.setAttribute('id', 'leftHalf');
            leftHalf.setAttribute('d', leftTrapPath);
            leftHalf.setAttribute('fill', 'url(#left_' + id + ')');
            leftHalf.setAttribute('filter', 'url(#BackFilter_' + id + ')');
            svg._svg.appendChild(leftHalf);


            var rightTrapPath = `M${center - smooth},0 L${center - smooth},${height} L${rightBot - cutThick + smooth},${height} L${rightTop - cutThick + smooth},0`;
            rightHalf.setAttribute('id', 'rightHalf');
            rightHalf.setAttribute('d', rightTrapPath);
            rightHalf.setAttribute('fill', 'url(#right_' + id + ')');
            rightHalf.setAttribute('filter', 'url(#BackFilter_' + id + ')');
            svg._svg.appendChild(rightHalf);


            //cut imitation //////////////////
            var leftLine = document.createElementNS('http://www.w3.org/2000/svg', 'line');
            var rightLine = document.createElementNS('http://www.w3.org/2000/svg', 'line');

            leftLine.setAttribute('x1', leftTop + cutThick / 2 - smooth);
            leftLine.setAttribute('y1', -cutThick / 10);
            leftLine.setAttribute('x2', leftBot + cutThick / 2);
            leftLine.setAttribute('y2', height + cutThick / 10); //for cut imitation, edges pass behind viewbox 
            leftLine.setAttribute('style', `stroke:${cutColor};stroke-width:${cutThick}`);
            leftLine.setAttribute('class', 'cutImitation');

            svg._svg.appendChild(leftLine);

            rightLine.setAttribute('x1', rightTop - cutThick / 2);
            rightLine.setAttribute('y1', -cutThick / 10);
            rightLine.setAttribute('x2', rightBot - cutThick / 2);
            rightLine.setAttribute('y2', height + cutThick / 10);
            rightLine.setAttribute('style', `stroke:${cutColor};stroke-width:${cutThick}`);
            rightLine.setAttribute('class', 'cutImitation');
            svg._svg.appendChild(rightLine);
            //////

            //gradient vectors
            var leftX1, leftX2, leftY1, leftY2;
            var rightX1, rightX2, rightY1, rightY2;

            leftX1 = (leftBot + leftTop + 2 * cutThick) / 2;//
            leftX2 = center; //gradient vectors meeting in the same point
            leftY1 = height / 2;
            leftY2 = (leftX1 * leftX1 + leftY1 * leftY1 + (leftTop + cutThick) * center - (leftTop + cutThick) * leftX1
                - 0 - leftX1 * leftX2) / (leftY1);
            rightX1 = (rightBot + rightTop - 2 * cutThick) / 2;
            rightX2 = leftX2;
            rightY1 = leftY1;
            rightY2 = leftY2;

            var leftVector = {
                y1: leftY1 / height * 100, //to %
                x1: leftX1 / wide * 100, ///2
                y2: leftY2 / height * 100,
                x2: leftX2 / wide * 100 + 10,//+ //to remove a 'triangle', the meeting point must to be calced a little deeper
            };

            var rightVector = {
                y1: rightY1 / height * 100,
                x1: rightX1 / wide * 100,
                y2: rightY2 / height * 100,
                x2: rightX2 / wide * 100 - 10,//-
            };
            this.setGradient('left', leftVector, svg);
            this.setGradient('right', rightVector, svg);
        },

        setGradient: function (id, vector, svg) {
            var gradient = document.createElementNS('http://www.w3.org/2000/svg', 'linearGradient');

            var gradStop1 = document.createElementNS('http://www.w3.org/2000/svg', 'stop');
            var gradStop2 = document.createElementNS('http://www.w3.org/2000/svg', 'stop');
            var gradStop3 = document.createElementNS('http://www.w3.org/2000/svg', 'stop');
            var gradStop4 = document.createElementNS('http://www.w3.org/2000/svg', 'stop');
            var defs = document.createElementNS('http://www.w3.org/2000/svg', 'defs');
            var filter = document.createElementNS('http://www.w3.org/2000/svg', 'filter');
            var colorMatrix = document.createElementNS('http://www.w3.org/2000/svg', 'feColorMatrix');

            var id = id + '_' + svg._container.id;
            gradient.setAttribute('id', id);
            gradient.setAttribute('x1', vector.x1 + '%');
            gradient.setAttribute('y1', vector.y1 + '%');
            gradient.setAttribute('x2', vector.x2 + '%');
            gradient.setAttribute('y2', vector.y2 + '%');
            gradient.setAttribute('gradientUnits', 'userSpaceOnUse');


            gradStop1.setAttribute('offset', 0);
            gradStop1.setAttribute('style', "stop-color:#686868");
            gradient.appendChild(gradStop1);

            gradStop2.setAttribute('offset', 0.18);
            //chosen offset slightly less, than same offset in tube.svg document
            //same reason, remove 'triangle'
            gradStop2.setAttribute('style', "stop-color:#999999");
            gradient.appendChild(gradStop2);

            gradStop3.setAttribute('offset', 0.95);
            gradStop3.setAttribute('style', "stop-color:#C7C7C7");
            gradient.appendChild(gradStop3);

            filter.setAttribute('id', 'BackFilter_' + svg._container.id);
            colorMatrix.setAttribute('id', 'feColorMatrix');

            colorMatrix.setAttribute('values', this.mvc.model.get(Enums.ParameterRoles.FILL_COLOR));
            filter.appendChild(colorMatrix);
            defs.appendChild(filter);
            svg._svg.appendChild(defs);

            svg._svg.appendChild(gradient);
        },
    });

    return TruncatedConeController;
});