﻿// <copyright file="DataAdapter.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>
// 


define(['server/Server', 'common/Utilites',
    'server/adapters/DataAdapterErrorProcessor', 'common/Error', 'common/Enums', 'helpers/PermissionChecker'], function (server, utils, DataAdapterErrorProcessor, Error, enums, PermissionChecker) {

        var DataAdapter = Class.extend({
            stateVars: enums.serverState,

            init: function (serverStateManager) {
                this.serverStateManager = serverStateManager;
                this._getAllResponseBind = this._getAllResponse.bind(this);
                this.sGet = this.serverStateManager.get.bind(this.serverStateManager);
                this.sSet = this.serverStateManager.set.bind(this.serverStateManager);
                this.errorProcessor = new DataAdapterErrorProcessor(this.serverStateManager);
                this.serverStateManager.subscribePropertyChanged(this.stateChanged, this);
                PermissionChecker.setDataAdapter(this);
            },

            stateChanged: function (event) {
            },

            initialize: function () {

                if (this.sGet(this.stateVars.SubscriptionId) === -1) {
                    this.sSet(this.stateVars.ackSequenceNumber, 0);
                    return this.createDataSubscription();
                }

                return Promise.resolve();
            },
            initializeHistory: function () {
                if (this.sGet(this.stateVars.SubscriptionId) === -1) {
                    this.sSet(this.stateVars.ackSequenceNumber, 0);
                    return this.createHistoryDataSubscription();
                }
                return Promise.resolve();
            },

            addMonitoredData: function (items) {
                var query = {
                    subscriptionId: this.sGet(this.stateVars.SubscriptionId),
                    items: items
                };

                return this.sendQuery(enums.queryType.createMonitoredDataItems, query);
            },

            refresh: function () {
                var query = {
                    subscriptionId: this.sGet(this.stateVars.SubscriptionId),
                };

                return this.sendQuery(enums.queryType.refreshData, query);
            },

            destroy: function () {
                if (this.sGet(this.stateVars.SubscriptionId) !== -1) {

                    var query = {
                        subscriptionId: this.sGet(this.stateVars.SubscriptionId)
                    }

                    //проставляем удаление без ожидания ответа
                    //т.к нет никакой необходимости ожидать возрат запроса            
                    this.sSet(this.stateVars.ackSequenceNumber, 0);
                    this.sSet(this.stateVars.SubscriptionId, -1, true);
                    return this.sendQuery(enums.queryType.deleteDataSubscription, query);
                }

                return Promise.resolve();
            },

            getAll: function () {
                var query = {
                    subscriptionId: this.sGet(this.stateVars.SubscriptionId),
                    ackSequenceNumber: this.sGet(this.stateVars.ackSequenceNumber)
                };
                return this.sendQuery(enums.queryType.publishData, query).then(this._getAllResponseBind);
            },

            _getAllResponse: function (result) {
                this._updateServerTime(result);

                if (result.data.subscriptionId == this.sGet(this.stateVars.SubscriptionId) && result.data.sequenceNumber && result.data.sequenceNumber > this.sGet(this.stateVars.ackSequenceNumber)) {
                    this.sSet(this.stateVars.ackSequenceNumber, result.data.sequenceNumber);
                }
                return result.data.recs || [];
            },

            writeData: function (items) {
                var query = { recs: items };
                return this.sendQuery(enums.queryType.writeData, query);
            },

            deleteMonitoredEvent: function (monitoredItemsId) {
                var query = {
                    subscriptionId: this.sGet(this.stateVars.SubscriptionId),
                    items: monitoredItemsId instanceof Array ? monitoredItemsId : [monitoredItemsId]
                };

                return this.sendQuery(enums.queryType.deleteMonitoredDataItems, query);
            },

            getLoginData: function () {
                return this.sendQuery(enums.queryType.getLoginData, '');
            },

            login: function (login, password) {
                return this.sendQuery(enums.queryType.login, { login: login, password: password });
            },

            getState: function (ip) {
                return server.sendQuery(enums.queryType.getState, '', ip).then(function (result) {
                    return result;
                }, function (result) {
                    result = {};
                    result.command = "getState";
                    result.data = {};
                    result.data.isMaster = false;
                    result.data.url = ip;
                    return Promise.resolve(result);
                });
            },

            logout: function () {
                return this.sendQuery(enums.queryType.logout, '').then(function (result) {
                    location.reload();
                }, function (result) {
                    location.reload();
                });
            },

            sendQuery: function (command, parameters, url) {
                var ctx = this;
                return server.sendQuery(command, parameters, url).then(function (result) {
                    if (url) {
                        result.url = url;
                    }
                    ctx.processResponse(result);
                    return result;
                }, function (result) {
                    return Promise.reject(ctx.processResponse(result));
                });
            },

            processResponse: function (result) {
                if (result.status == 200 && !result.data.redirect && result.data.code == 0) {
                    this.errorProcessor.resetErrorStatus(result.command);
                }
                else {
                    if (result.status == 501 || result.status == 500 || result.status == 503 || result.status == 0 || result.data.code > 0x80000000) {
                        return this.errorProcessor.handleServerError(result);
                    }
                }
            },
            publishHistoryData: function (storage, requestId, params) {
                var query = {
                    subscriptionId: this.sGet(this.stateVars.SubscriptionId),
                    requestId: requestId
                };
                if (params) {
                    query = _.assign(query, params);
                }
                var ctx = this;
                return this.sendQuery(enums.queryType.publishHistoryData, query).then(function (result) {
                    if (result.data.data) {
                        for (var i = 0; i < result.data.data.length; i++) {
                            if (storage[result.data.data[i].archiveItemId] === undefined)
                                storage[result.data.data[i].archiveItemId] = [];
                            storage[result.data.data[i].archiveItemId] = storage[result.data.data[i].archiveItemId].concat(result.data.data[i].values);
                        }
                    }
                    if (result.data.code) {
                        switch (result.data.code) {
                            case 0x80320000: //запрос в обработке, но не готов
                                return ctx.publishHistoryData(storage, requestId);
                            case 0x802A0000: //запрос был удален, нужно отправить полученный requestId
                                if (result.data.requestId > requestId)
                                    return ctx.publishHistoryData(storage, result.data.requestId);
                            case 0x00A60000:
                                return ctx.publishHistoryData(storage, requestId);
                            case 0x80840000:
                                return Promise.reject(ctx.processResponse(result));
                            case 0x804A0000: //нет запрашиваемого subscriptionId
                            default:
                                return storage;
                        }
                    }
                    return storage;
                });
            },
            createDataSubscription: function () {
                var query = {
                    requestedPublishingInterval: this.sGet(this.stateVars.requestedPublishingInterval),
                    requestedLifetimeInterval: this.sGet(this.stateVars.requestedLifetimeInterval),
                    maxNotificationsPerPublish: this.sGet(this.stateVars.maxNotificationsPerPublish),
                    maxSize: this.sGet(this.stateVars.maxSize)
                }

                var ctx = this;
                return this.sendQuery(enums.queryType.createDataSubscription, query).then(function (result) {
                    ctx._updateServerTime(result);
                    ctx.sSet(ctx.stateVars.SubscriptionId, result.data.subscriptionId);
                    ctx.sSet(ctx.stateVars.requestedPublishingInterval, result.data.revisedPublishingInterval);
                    ctx.sSet(ctx.stateVars.requestedLifetimeInterval, result.data.revisedLifetimeInterval);
                    ctx.sSet(ctx.stateVars.Connected, true);
                });

            },
            createHistoryDataSubscription: function () {
                var query = {
                    //                requestedPublishingInterval: this.sGet(this.stateVars.requestedPublishingInterval),
                    requestedLifetimeInterval: this.sGet(this.stateVars.requestedLifetimeInterval),
                    maxNotificationsPerPublish: this.sGet(this.stateVars.maxNotificationsPerPublish),
                    maxSize: this.sGet(this.stateVars.maxSize)
                }
                var ctx = this;
                return this.sendQuery(enums.queryType.createHistoryDataSubscription, query).then(function (result) {
                    ctx._updateServerTime(result);
                    ctx.sSet(ctx.stateVars.SubscriptionId, result.data.subscriptionId);
                    //                ctx.sSet(ctx.stateVars.requestedPublishingInterval, result.data.revisedPublishingInterval);
                    ctx.sSet(ctx.stateVars.requestedLifetimeInterval, result.data.revisedLifetimeInterval);
                    ctx.sSet(ctx.stateVars.Connected, true);
                });
            },
            getArchiveItems: function (items) {
                var query = {
                    items: items
                };

                return this.sendQuery(enums.queryType.getArchiveItems, query).then(function (result) {
                    this._updateServerTime(result);
                    return result.data.items || [];
                }.bind(this));
            },
            getAsyncArchiveData: function (params, items) {
                var parameters = $.extend({
                    startTime: 0,
                    endTime: 0,
                    numValuesPerNode: 0,
                    returnBounds: true,
                    returnFirstLastValue: false
                }, params);
                if ((parameters.startTime < 0 || parameters.endTime < 0) || (!parameters.returnFirstLastValue && parameters.startTime == 0)) {
                    return Promise.reject("Invalid request getArchiveData: " + JSON.stringify(parameters));
                }
                var query = {
                    parameters: parameters,
                    data: items
                };
                var ctx = this;
                return this.sendQuery(enums.queryType.historyReadRawAsync, query).then(function (result) {
                    return ctx.publishHistoryData({}, result.data.requestId).then(function (result) {
                        return result || [];
                    });
                });
            },
            getArchiveData: function (params, items) {
                var parameters = $.extend({
                    startTime: 0,
                    endTime: 0,
                    numValuesPerNode: 0,
                    returnBounds: true,
                    returnFirstLastValue: false
                }, params);
                if ((parameters.startTime < 0 || parameters.endTime < 0) || (!parameters.returnFirstLastValue && parameters.startTime == 0)) {
                    return Promise.reject("Invalid request getArchiveData: " + JSON.stringify(parameters));
                }
                var query = {
                    parameters: parameters,
                    data: items
                };
                return this.sendQuery(enums.queryType.historyReadRaw, query).then(function (result) {
                    return result.data.data || [];
                });
            },

            _updateServerTime: function (result) {
                if (result.data && result.data.serverTime > this.sGet(this.stateVars.serverTime)) {
                    this.sSet(this.stateVars.serverTime, Number(result.data.serverTime));
                }
            },

            reportAction: function (items) {
                var query = {
                    data: items
                }
                return this.sendQuery(enums.queryType.reportAction, query);
            },
        });

        return DataAdapter;
    });