﻿

/*
* @class EventTarget
* This class contains functionality 
*/
define(['common/Error'], function (Error) {

    var undefined;

    var EventTarget = Class.extend({
        init: function () {
            this._listeners = {};
        },

        /*
        * @function 
        * @name addListener
        * @param type {string} property name
        * @param listener {function} function (event{target, type, property}){}
        * @param context (object) context will be 'this' in handler
        */
        addListener: function (type, listener, context) {
            if (this._listeners[type] === undefined) {
                this._listeners[type] = [];
            }

            this._listeners[type].push({ handler: listener, context: context });
        },

        bindOnce: function (type, listener, context) {
            if (this._listeners[type] === undefined) {
                this._listeners[type] = [];
            }

            this._listeners[type].push({
                handler: function (ctx, i) {
                    return function (event) {
                        listener.call(context,event);
                        if (ctx._listeners[type] instanceof Array) {
                            ctx._listeners[type].splice(i, 1);
                        }
                    }
                }(this, this._listeners[type].length),
                context: context
            });
        },

        /*
        * @function
        * @name fire
        * @param event event{target, type, property} 
        */
        fire: function (event) {
            if (typeof event == "string") {
                event = { type: event };
            }

            if (!event.type) {  
                throw new Error("Event object missing 'type' property.");
            }

            if (this._listeners[event.type] instanceof Array) {
                var listeners = this._listeners[event.type];
                for (var i = 0; i < listeners.length; i++) {
                    try {
                        listeners[i].handler.call(listeners[i].context, event);
                    }
                    catch (e) {
                        Error.exception(e);
                    }
                }
            }
        },

        /*
        * @function 
        * @name removeListener
        * @param type {string} property name
        * @param listener {function} function (event{target, type, property}){}
        */
        removeListener: function (type, listener) {
            if (this._listeners[type] instanceof Array) {
                var listeners = this._listeners[type];
                for (var i = 0; i < listeners.length; i++) {
                    if (listeners[i].handler === listener) {
                        listeners.splice(i, 1);
                        break;
                    }
                }
            }
        },

        removeAllListeners: function (type) {
            if (this._listeners[type] instanceof Array) {
                this._listeners[type] = [];               
            }
        }

    });

    return EventTarget;
});