﻿// <copyright file="JournalController.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>
// 

define(['when', 'base/ControlController',
    'common/Enums', 'common/Utilites',
    'controls/Journal/modules/MessageHolder',
    'controls/Journal/modules/details/DetailsPanel',
    'controls/Journal/modules/filter/FilterPanel',
    'controls/Journal/modules/grid/Grid',
    'controls/Journal/modules/toolbar/Toolbar',
     'server/requestManager/RequestManager',
     'controls/Journal/modules/ToggleFullScreen',
     'controls/Journal/modules/BottomBar',
    'helpers/PathBuilder',
    'common/Appearence',
    'core/convertations/ConvertationChain',
    'core/ImplicitConvertations',
    'link!controls/Journal/styles/grid.css',
    'link!controls/Journal/styles/layout.css',
],
    function (when, ControlController, Enums, Utilites, MessageHolder, Details,
        FilterPanel, Grid, ToolBar, RequestManager, ToggleFullScreen,
        BottomBar, PathBuilder, Appearence, ConvertationChain, ImplicitConvertations) {

        var JournalController = ControlController.extend({
            gridHorizontalScrollHeight: 15,
            minHeight: 800,
            toolbarHeight: 50,
            bottomBarHeight: 25,
            init: function () {
                this._super();
                this.initialized = false;
                this.visualPartsCreated = false;
                this.ClassName = 'JournalController';
                this.archiveLeftBorder = 0;
                this.isFromFirstPage = false;
            },

            validateProperty: function (propertyName, newProperty) {
                switch (propertyName) {
                    case Enums.ParameterRoles.WIDTH:
                        this.validator.validateMinMax(
                            {
                                propertyName: propertyName,
                                minValue: this.minHeight,
                                property: newProperty
                            });
                        break;
                }
            },

            onParentActualWidthChanged: function (event) {
                this._super(event);
                // this.updateHeights();
            },

            onParentActualHeightChanged: function (event) {
                this._super(event);
                this.updateHeights();
            },

            onAddedToDOM: function () {
                this._super();
                this._cacheConverters();

                this.requestManager = this._createRequestManager();
                var objectUid = this.mvc.model.get(Enums.ParameterRoles.OBJECT_UID);
                if (objectUid != "") {
                    this.requestManager.setObject(objectUid);
                } else if (!this.mvc.model.get(Enums.ParameterRoles.IS_GLOBAL)) {
                    this.requestManager.setObject(this.getParentObject());
                }
                this.messageHolder = new MessageHolder(this.mvc.model, this.requestManager);
                this.messageHolder.useArchive = this.mvc.model.get(Enums.ParameterRoles.USE_ARCHIVE);
                if (this.mvc.model.get(Enums.ParameterRoles.IS_VISIBLE)) {
                    this._createVisualParts();
                }

                this.initialized = true;
                this.onIsVisibleChanged(this.mvc.model.get(Enums.ParameterRoles.IS_VISIBLE));
                if (this.mvc.model.getAutoScroll() && !this.grid.isDefaultSortOrder()) {
                    //если включен автоскролл, но сортировка задана не по времени 
                    //то отключаем его
                    this.mvc.model.set(Enums.ParameterRoles.AUTOSCROLL, false);
                }

                this.startPooling();

                if (!this.mvc.model.getAutoScroll()) {
                    this.messageHolder.loadArchive(true);
                }
                this.pageChanged = false;
            },

            _createVisualParts: function () {
                this.visualPartsCreated = true;
                this.grid = new Grid(this.messageHolder, this.mvc.model);
                this.toolBar = new ToolBar(this.mvc.model);
                this.toolBar.subscribeActionCalled(this._onActionCalled, this);
                this.bottomBar = new BottomBar(this.messageHolder, this.mvc.model);

                this.filter = new FilterPanel(this.mvc.model);
                this.details = new Details(this.messageHolder, this.mvc.model);

                this.toolBar.render(this.mvc.view.$());
                this.filter.render(this.mvc.view.$());
                this.details.render(this.mvc.view.$());
                this.bottomBar.render(this.mvc.view.$());

                this.grid.initialize();
                this.grid.render(this.mvc.view.$());
                this.grid.stylize();
                this.grid.afterRender(this._getContainerHeight());

                this._onShowGridChanged();
                if (this.mvc.model.getFullScreen()) {
                    this._onFullScreenChanged();
                }
                else {
                    this.updateHeights();
                }
            },

            _createRequestManager: function () {
                var requestManager = new RequestManager(this.mvc.model.getOptionsForEM());
                requestManager.setColumns(this.mvc.model.getAllColumnsForEM());
                requestManager.setFilters(this.mvc.model.getFiltersForEM(), true);

                return requestManager;
            },

            updateHeights: function () {
                if (this.initialized === true && this.visualPartsCreated == true) {
                    var containerHeight = this._getContainerHeight();
                    var detailsHeight = 0;


                    if (this.mvc.model.getShowLegend()) {
                        detailsHeight = this.details.getHeight();

                        if (this.mvc.model.getShowFilter()) {
                            if (containerHeight < (detailsHeight + this.filter.getHeight())) {
                                detailsHeight = containerHeight - this.filter.getHeight();
                            }
                        }

                        // не даём details занять всё свободное место
                        if ((detailsHeight / containerHeight) > 0.7) {
                            detailsHeight = 0.7 * containerHeight;
                        }

                        this.details.setHeight(detailsHeight);
                    }
                    this.grid.setHeight(containerHeight - detailsHeight);
                }
            },

            _getContainerHeight: function () {
                return this.mvc.view.$().height() - (this.mvc.model.isToolBarEnabled() ? this.toolbarHeight : 0)
                    - this.bottomBarHeight - this.gridHorizontalScrollHeight;
            },

            startPooling: function () {
                return this.initArchive().then(function () {
                    this.requestManager.subscribe();
                }.bind(this));
            },

            initArchive: function () {
                if (this.mvc.model.get(Enums.ParameterRoles.USE_ARCHIVE)) {
                    return this.messageHolder.getArchiveLeftBorder().then(function (timestamp) {
                        this.archiveLeftBorder = timestamp;
                    }.bind(this));
                } else {
                    return when.resolve();
                }
            },

            restartPooling: function () {
                var promise;
                promise = this.messageHolder.loadArchiveByInterval(this.mvc.model.getRightBorder() - this.mvc.model.getInterval(), this.mvc.model.getRightBorder());
                return promise.then(function () {
                    this.startPooling();
                }.bind(this));
            },

            modelPropertyChangedInternal: function (event) {
                this._super(event);
                //обработается до дочерних модулей, т.к подписана первой            
                switch (event.property) {
                    case Enums.ParameterRoles.LEGEND_BACKGROUND_COLOR:
                        this._LegendBackgroundColorChanged();
                        break;
                    case Enums.ParameterRoles.TOOLBAR_BACKGROUND_COLOR:
                        this._onToolbarBackgroundColorChanged();
                        break;
                    case Enums.ParameterRoles.FILTERS_BACKGROUND_COLOR:
                        this._FiltersBackgroundColorChanged();
                        break;
                }

                if (this.initialized === true) {
                    switch (event.property) {
                        case Enums.ParameterRoles.AUTOSCROLL:
                            this._onAutoScrollChanged();
                            break;
                        case Enums.ParameterRoles.SHOW_GRID:
                            this._onShowGridChanged();
                            break;
                        case Enums.ParameterRoles.SHOW_FILTER:
                            //this.updateHeights();
                            break;
                        case Enums.ParameterRoles.SHOW_LEGEND:
                            this._onShowLegendChanged();
                            break;
                        case Enums.ParameterRoles.SHOW_TOOLBAR:
                            this._onShowToolbarChanged();
                            break;
                        case Enums.ParameterRoles.FULL_SCREEN:
                            this._onFullScreenChanged();
                            break;
                        case Enums.ParameterRoles.WIDTH:
                        case Enums.ParameterRoles.HEIGHT:
                            this._onParentSizeChanged();
                            break;
                        case Enums.ParameterRoles.TOOLBAR_ENABLED:
                            this._onToolBarEnabledChanged(event);
                            break;
                        case Enums.ParameterRoles.INTERVAL:
                            this._onIntervalChanged();
                            break;
                        case Enums.ParameterRoles.TILL:
                            this._onTillChanged();
                            break;
                        case Enums.ParameterRoles.OBJECT_UID:
                            this._onObjectUIDChanged();
                            break;
                        default:
                            var pathArray = new PathBuilder(event.property).getArray();
                            if (pathArray[0] === Enums.ParameterRoles.FILTERS
                                && Utilites.isNumber(pathArray[1])
                                && pathArray[2] === Enums.ParameterRoles.ENABLED) {
                                this._onFilterChanged();
                            }

                            break;
                    }
                }

            },
            _onObjectUIDChanged: function () {
                this.requestManager.setObject(this.mvc.model.get(Enums.ParameterRoles.OBJECT_UID));
                var ctx = this;
                return this.requestManager.deleteMonitoredEvent().then(function () {
                    ctx.startPooling();
                });
            },
            _onIntervalChanged: function () {
                if ((this.mvc.model.getInterval() <= 0) || (!this.archiveLeftBorder)  || !this.messageHolder.useArchive)
                    return;
                if (this.mvc.model.getAutoScroll()) {
                    this.mvc.model.stopAutoscroll(Enums.autoscrollStopReason.pageChange);
                    console.log('INTERVAL!!111');
                }
                var interval = this.mvc.model.getInterval();
                var leftBorder = this.mvc.model.getLeftBorder();
                var rightBorder = this.mvc.model.getRightBorder();
                //если расширяем интервал от конца архива
                if (this.isFromFirstPage) {
                    this.mvc.model.setRightBorder(leftBorder + interval);
                    if (leftBorder + interval < rightBorder) { //интервал уменьшился, нужно удалить сообщения которые в него больше не входят
                        this.messageHolder.deleteOldMessaged(false);
                        this.grid.selectFirstMessage();
                    } else { //интервал увеличился, нужно загрузить недостающие сообщения
                        leftBorder = rightBorder + 1;
                        rightBorder = this.archiveLeftBorder + interval;
                        return this.messageHolder.loadArchiveByInterval(leftBorder, rightBorder);
                    }
                } else { //расширямся от начала
                    this.mvc.model.setLeftBorder(rightBorder - interval);
                    if (leftBorder + interval < rightBorder) { //интервал уменьшился, нужно удалить сообщения которые в него больше не входят
                        this.messageHolder.deleteOldMessaged(true);
                        this.grid.selectFirstMessage();
                    } else { //интервал увеличился, нужно загрузить недостающие сообщения
                        rightBorder = leftBorder - 1;
                        leftBorder = rightBorder - interval;
                        return this.messageHolder.loadArchiveByInterval(leftBorder, rightBorder);
                    }
                }

            },
            _onTillChanged: function () {
                if (!this.mvc.model.getAutoScroll() && this.messageHolder.useArchive) { //если включена автопрокрутка, то параметр конец не обрабатывается
                    if (this.pageChanged) { //конец изменился вследствие листания
                        this.pageChanged = false;
                        return;
                    }
                    var interval = this.mvc.model.getInterval();
                    var rightBorder = this.mvc.model.get(Enums.ParameterRoles.TILL);
                    var leftBorder = rightBorder - interval;;
                    this.mvc.model.setRightBorder(rightBorder);
                    this.mvc.model.setLeftBorder(rightBorder - interval);
                    this.messageHolder.deleteAll();
                    return this.messageHolder.loadArchiveByInterval(leftBorder, rightBorder);
                }
            },
            _onToolBarEnabledChanged: function (event) {
                var cssValue = this.mvc.model.isToolBarEnabled() ? '' : 'none';

                this.mvc.view.$('[placeholder="toolbar"]').css('display', cssValue);
            },

            onIsVisibleChanged: function (value) {
                this._super(value);
                var visibility = this.mvc.model.get(Enums.ParameterRoles.IS_VISIBLE);
                if (visibility == true && this.initialized === true && this.visualPartsCreated == false) {
                    this._createVisualParts();
                }
            },

            _onParentSizeChanged: function () {
                if (this.visualPartsCreated) {
                    this.updateHeights();
                }
            },

            _onAutoScrollChanged: function () {
                if (this.mvc.model.getAutoScroll()) {
                    this.mvc.model.setRightBorder(this.messageHolder.currentTime);
                    this.mvc.model.setLeftBorder(this.messageHolder.currentTime - this.mvc.model.getInterval());
                    /*if (!this.messageHolder.useArchive) {

                    } else {*/
                        this._startRightBorderUpdate();
                    //}
                }
            },

            _startRightBorderUpdate: function () {
                this.messageHolder.deleteAll();
                //перед стартом запрашиваем сообщения с последней страницы
                return this.messageHolder.loadArchiveByInterval(this.mvc.model.getLeftBorder(), this.mvc.model.getRightBorder());
            },
            _onShowLegendChanged: function () {
                if (this.visualPartsCreated) {
                    this.updateHeights();
                }
            },

            _onShowToolbarChanged: function () {
                if (this.visualPartsCreated) {
                    this.updateHeights();
                }
            },

            _onShowGridChanged: function () {
                if (this.mvc.model.getShowGrid()) {
                    this.grid.show();
                } else {
                    this.grid.hide();
                }

                if (this.visualPartsCreated) {
                    this.updateHeights();
                }
            },

            _onFullScreenChanged: function () {
                if (this.visualPartsCreated) {
                    var isFullScreen = this.mvc.model.getFullScreen();
                    ToggleFullScreen(this.mvc.view.$(), isFullScreen);
                    this.updateHeights();
                }
            },

            _onFilterChanged: function () {
                var ctx = this;
                return this.requestManager.deleteMonitoredEvent().then(function (result) {
                    return ctx.requestManager.setFilters(ctx.mvc.model.getFiltersForEM(), true).then(function () {
                        ctx.messageHolder.clear();
                        return ctx.restartPooling();
                    }).then(function () {
                        if (ctx.messageHolder.selectedMessage) {
                            ctx.grid.selectRow(ctx.messageHolder.selectedMessage[ctx.messageHolder.keyFieldName],
                                true);
                        }
                    });
                });
            },

            _onActionCalled: function (event) {
                if (this[event.action] !== undefined) {
                    this[event.action]();
                }
            },

            callMethod: function (methodName, params) {
                return this[methodName].apply(this, params);
            },

            firstPage: function () {
                if (this.messageHolder.useArchive) {
                    this.messageHolder.deleteAll();
                    if (!this.archiveLeftBorder) {
                        return this.messageHolder.getArchiveLeftBorder().then(function (timestamp) {
                            if (timestamp) {
                                this.archiveLeftBorder = timestamp;
                                return this.loadMessagesFromFirstPage();
                            }
                        }.bind(this));
                    } else {
                        return this.loadMessagesFromFirstPage();
                    }
                }
            },

            loadMessagesFromFirstPage: function () {
                this.isFromFirstPage = true;
                this.mvc.model.stopAutoscroll(Enums.autoscrollStopReason.pageChange);
                //сдвигаем обе границы на левый край, после загрузки страницы
                //правая граница верно выравняется
                this.messageHolder.clear();
                this.mvc.model.setLeftBorder(this.archiveLeftBorder - 1);
                this.mvc.model.setRightBorder(this.archiveLeftBorder + this.mvc.model.getInterval());
                return this.messageHolder.loadArchiveByInterval(this.mvc.model.getLeftBorder(), this.mvc.model.getRightBorder()).then(function () {
                    this.grid.selectLastMessage();
                }.bind(this));
            },
       
            getRowScrollerPosition: function () { //на каком сообщении из контейнера находится сейчас скролл
                if (this.grid.table.scroll.max)
                    return Math.ceil(this.grid.table.scroll.sliderPosition / this.grid.table.scroll.max * this.messageHolder._getMessageCount());
                else
                    return 0;
            },
            getMessagesPageLackFromStart: function () {
                if (this.messageHolder._getMessageCount() < this.grid.tbody.rows.length || !this.grid.scroll.visible) {
                    return this.grid.tbody.rows.length;
                }
                if (this.getRowScrollerPosition() < this.grid.tbody.rows.length)
                    return (this.grid.tbody.rows.length - this.getRowScrollerPosition());
                else
                    return 0;
            },
            getMessagesPageLackFromEnd: function () {
                if (this.messageHolder._getMessageCount() == this.getRowScrollerPosition() || !this.grid.scroll.visible) {
                    return this.grid.tbody.rows.length;
                }
                if (this.messageHolder._getMessageCount() - this.getRowScrollerPosition() < this.grid.tbody.rows.length)
                    return (this.messageHolder._getMessageCount() - this.getRowScrollerPosition());
                else
                    return 0;
            },
            isActiveFieldSort: function(){
                return (this.grid.currentSortField === this.grid.activeTimeField);
            },
            isSortOrderAsc: function () {
                return (this.grid.currentSortOrder === Enums.SortType.Asc);
            },
            previousMessage: function () {
                this.grid.selectNextMessage();
                return when.resolve();
            },

            nextMessage: function () {
                this.grid.selectPreviousMessage();
                return when.resolve();
            },

            previousPage: function () {
                if (!this.archiveLeftBorder || !this.messageHolder.useArchive) //сообщений нет
                    return when.resolve();
                this.mvc.model.stopAutoscroll(Enums.autoscrollStopReason.pageChange);
                this.pageChanged = true;
                console.log("Сортировка по возр.: " + this.isSortOrderAsc() + " Не хватает с конца: " + this.getMessagesPageLackFromEnd() + " Не хватает с начала: " + this.getMessagesPageLackFromStart());
                if ((this.isSortOrderAsc() && this.getMessagesPageLackFromStart()) || (!this.isSortOrderAsc() && this.getMessagesPageLackFromEnd()) && !this.isFromFirstPage) {
                    if (this.isActiveFieldSort()) {
                        //    this.isFromFirstPage = false;
                        var numOfMessages;
                        if (this.isSortOrderAsc()) {
                            numOfMessages = this.getMessagesPageLackFromStart();
                        } else {
                            numOfMessages = this.getMessagesPageLackFromEnd();
                        }
                        this.messageHolder.requestManager.getArchiveEvents(0, this.mvc.model.getLeftBorder() - 1, numOfMessages, true).then(function (messages) {
                            console.log("Сервер вернул " + messages.length + " сообщений");
                            if (messages && messages.length > 0) {
                                if (messages.length < this.getMessagesPageLackFromEnd()) { //сообщений пришло меньше чем запрашивали, значит мы достигли начала журнала
                                    this.isFromFirstPage = true;
                                }
                                this.mvc.model.setLeftBorder(this.messageHolder.getLastMessageTime());
                                this.messageHolder.deleteOldMessaged(false);
                                this.grid.scroll.setMax(this.messageHolder._getMessageCount() - this.grid.tbody.rows.length);
                                this.grid.scroll.scrollToEnd();
                                this.grid.selectLastMessage();
                            }
                        }.bind(this));
                    }
                //    this.grid.scroll.scrollToStart();
                }
                else {
                    this.isFromFirstPage = false;
                    this.grid.scroll.changePosition(-this.grid.tbody.rows.length);
                }
                return when.resolve();
            },
                        
            nextPage: function () {
                if (!this.archiveLeftBorder || !this.messageHolder.useArchive) //сообщений нет
                    return when.resolve();
                this.mvc.model.stopAutoscroll(Enums.autoscrollStopReason.pageChange);
                this.pageChanged = true;
                this.isFromFirstPage = false;
                console.log("Сортировка по возр.: " + this.isSortOrderAsc() + " Не хватает с конца: " + this.getMessagesPageLackFromEnd() + " Не хватает с начала: " + this.getMessagesPageLackFromStart());
                if ((this.isSortOrderAsc() && this.getMessagesPageLackFromEnd()) || (!this.isSortOrderAsc() && this.getMessagesPageLackFromStart())) {
                    if (this.isActiveFieldSort()) {
                        var numOfMessages;
                        if (this.isSortOrderAsc()) {
                            numOfMessages = this.getMessagesPageLackFromEnd();
                        } else {
                            numOfMessages = this.getMessagesPageLackFromStart();
                        }
                        this.messageHolder.requestManager.getArchiveEvents(this.mvc.model.getRightBorder() + 1, 0, numOfMessages).then(function (messages) {
                            console.log("Сервер вернул " + messages.length + " сообщений");
                            if (messages && messages.length > 0) {
                                this.mvc.model.setRightBorder(this.messageHolder.getFirstMessageTime());
                                this.messageHolder.deleteOldMessaged(true);
                                this.grid.scroll.setMax(this.messageHolder._getMessageCount() - this.grid.tbody.rows.length);
                                this.grid.scroll.scrollToStart();
                                this.grid.selectFirstMessage();
                            }
                        }.bind(this));
                    }
                    this.grid.scroll.scrollToStart();
                } else {
                    this.grid.scroll.changePosition(+this.grid.tbody.rows.length);
                }
                return when.resolve();
            },

            lastPage: function () {
                //запуск автоскролла обновит границы
                if (this.messageHolder.useArchive) {
                    this.mvc.model.startAutoscroll();
                    this.isFromFirstPage = false;
                    return when.resolve();
                }
            },

            refresh: function () {
                this.requestManager.getAll();
                return when.resolve();
            },
            exportEvent: function () {
                var blob = new Blob([this.formTextToExport()], { type: "text/plain;charset=utf-8" });
                saveAs(blob, "journal.txt");
            },
            formTextToExport: function () {
                var result="";
                var i, j;
                var msgs = this.messageHolder.messageContainer.messages;
                var columns = this.grid.columns.filter(this.columnToExport);
                for (i=0; i<columns.length; i++){
                    result += columns[i].name + ",";
                }
                result += "\n";
                for (i = 0; i < msgs.length; i++) {
                    for (j = 0; j < columns.length; j++) {
                        if (msgs[i][columns[j].field] instanceof Date) {
                            var format;
                            if (columns[j].formatString)
                                format = columns[j].formatString;
                            else
                                format = ImplicitConvertations.getDateTimeDefaultFormat();
                            result += String.format('{0:' + format + '}',  msgs[i][columns[j].field])+",";
                        } else {
                            result += msgs[i][columns[j].field] + ",";
                        }
                    }
                    result += "\n";
                }
                return result;
            },
            columnToExport: function(obj){
                if (obj.name != "" && obj.field != "Ack" && obj.field != "EventType" && obj.visibility == true) {
                    return true;
                } else {
                    return false;
                }
            },
            ackEvents: function (comment) {
                //квитирует сообщения на странице
                return this.messageHolder.confirmAllMessages(comment);
            },

            _LegendBackgroundColorChanged: function () {
                Appearence.background.apply(this.mvc.view.$('div[placeholder="details"]'),
                    this.mvc.model.get(Enums.ParameterRoles.LEGEND_BACKGROUND_COLOR));
            },

            _onToolbarBackgroundColorChanged: function () {
                Appearence.background.apply(this.mvc.view.$('div[placeholder="toolbar"]'),
                    this.mvc.model.get(Enums.ParameterRoles.TOOLBAR_BACKGROUND_COLOR));
            },

            _FiltersBackgroundColorChanged: function () {
                Appearence.background.apply(this.mvc.view.$('div[placeholder="filter"]'),
                       this.mvc.model.get(Enums.ParameterRoles.FILTERS_BACKGROUND_COLOR));
            },

            _cacheConverters: function () {
                this._cacheConvertersForProperty(Enums.ParameterRoles.ROW_STYLE);
            },

            _cacheConvertersForProperty: function (propName) {
                this.mvc.model.convertersCache[propName] = {};
                var propObj = this.mvc.model.get(propName);
                for (var subPropName in propObj) {
                    var subPropObj = propObj[subPropName];

                    if (subPropObj.metadata && subPropObj.metadata.Convertations) {
                        this.mvc.model.convertersCache[propName][subPropName]
                          = new ConvertationChain(subPropObj.metadata.Convertations);
                    }
                }            
            },
            onDestroy: function () {
                this._super();
                this.requestManager.unsubscribe();
                this.initialized = false;
                this.messageHolder.eventTarget.removeAllListeners(this.messageHolder.MESSAGE_SELECTED);
                this.messageHolder.messageContainer = null;
                this.messageHolder.skippedMessageContainer = null;
                this.messageHolder.eventTarget.removeAllListeners(this.messageHolder.MESSAGES_UPDATED);
                this.toolBar.unsubscribeActionCalled(this._onActionCalled);
                this.requestManager.eventTarget.removeAllListeners(this.requestManager.MESSAGES_CHANGED);
                this.requestManager.eventTarget.removeAllListeners(this.requestManager.ARCHIVE_MESSAGES_RECEIVED);
                this.requestManager.eventTarget.removeAllListeners(this.requestManager.TIME_UPDATED);
            }
        });

        return JournalController;
    });
