﻿define(['base/ObservableObject',
        'controls/Trend/modules/xAxis',
        'controls/Trend/modules/yAxis',
        'controls/Trend/modules/Grid',
        'common/Enums'], function (ObservableObject, xAxis, yAxis, Grid, Enums) {

            var defaults = {
                font: '12px Arial',
                titleFontName: 'Arial',
                titleFontSize: 18,
                titleTopBottomPadding: 12
            },

            dk = 0,

            events = Enums.TrendEvents,
            BackLayer = ObservableObject.extend({
                init: function (options) {
                    
                    this._super();
                    this.options = $.extend(true, {}, defaults);
                    this.model = options.model;
                    this.titleHeight = this.model.get(Enums.ParameterRoles.TITLE) ?
                        this.options.titleFontSize + this.options.titleTopBottomPadding : 0;

                    this.gridCanvas = options.canvasPool.requestCanvas()
                        .setSize(options.size)
                        .setNodeSize({ width: '100%', height: '100%' });

                    this.canvas = options.canvasPool.requestCanvas()
                        .setSize(options.size)
                        .setNodeSize({ width: '100%', height: '100%' });
                   
                   
                    this.grid = new Grid(this.gridCanvas, options.size, options.model);
                    this.xAxis = new xAxis(this.canvas, options.size, options.model, options.max);
                    this.xAxis._updateHeight();
                    this.yAxis = new yAxis(this.canvas, {
                        width: options.size.width,
                        height: options.size.height - this.xAxis.getDimensions().height - this.titleHeight
                    }, options.model);                   

                    this.afterInit();
                },

                getTitleHeight: function () {
                    return this.titleHeight;
                },

                onValueChanged: function (newTs) {
                    this.xAxis.options.max = newTs;
                    this._drawInternal(this.canvas, function () {
                        this.xAxis.redrawTicks();
                    });
                },

                afterInit: function () {
                    this._positionAll();
                    this._drawAll();
                },

                getInterval: function () {
                    return this.model.get(Enums.ParameterRoles.INTERVAL);
                },

                getYRange: function () {
                    return [this.yAxis.options.min, this.yAxis.options.max];
                },

                updateYAxis: function (settings) {
                    this.yAxis.update(settings);
                    this.afterInit();
                },

                _drawInternal: function (canvas, actions) {
                    canvas.beginRender();
                    actions.call(this);
                    canvas.endRender();
                },

                _positionAll: function () {
                    this.yAxis.position({ x: 0, y: this.titleHeight });
                    var yDims = this.yAxis.getDimensions();
                    this.grid.height = this.yAxis.getAxisLineLength();
                    this.grid.y = this.yAxis.getYWithoutArrow();
                    this.grid.x = this.yAxis.getMinMajorTickX()
                        + this.model.get(Enums.ParameterRoles.Y_MAJOR_TICK_WIDTH) / 2;
                    this.xAxis.position({ x: yDims.width, y: yDims.height + yDims.y });
                    var ticks = this._getGridTicks();
                    this.grid.width = this.xAxis.getAxisLineLength();
                    this.grid.position(ticks[0][0], ticks[0][1], ticks[1][0], ticks[1][1]);
                    ticks = null;
                },

                _drawTitle: function (bbox, dt) {
                    var text = this.model.get(Enums.ParameterRoles.TITLE);
                    if (!text) {
                        return;
                    }
                    var font = String.format('{0}px {1}', this.options.titleFontSize, this.options.titleFontName);                  
                    this.canvas.setFont(font);                    
                    var txtWidth = this.canvas.measureText(text);
                    var grid = this.canvas.getSize();
                    var x = (grid.width / 2) - (txtWidth / 2);
                    this.canvas.setFontStyle(font);
                    this.canvas.fillText({
                        text: text,
                        x: x,
                        y: 10,
                        maxWidth: grid.width
                    });
                    this.canvas.setFontStyle(this.options.font);
                    this.canvas.setFont(this.options.font);
                },

                _drawAll: function () {
                    dk++;
                    this.eventTarget.fire(events.beforeRender);
                    this._drawInternal(this.canvas, function () {
                        this.canvas.clear();

                        this.canvas.setFont(this.options.font);
                        // обновляем грид
                        // рисуем его до меток, чтобы соблюсти z-порядок (метки должны перекрывать грид)
                        this._drawInternal(this.gridCanvas, function () {
                            this.gridCanvas.clear();
                            this.grid.draw();
                        });

                        this.xAxis.draw();
                        this.yAxis.draw();
                        this._drawTitle();
                    });

                    this.eventTarget.fire(events.afterRender);
                },

                _getGridTicks: function () {
                    var majorTicksLines = this._getGridMajorTicks();
                    var minorTicksLines = this._getGridMinorTicks();
                    return [majorTicksLines, minorTicksLines];
                },
           
                _getGridMajorTicks: function () {
                    var xTh = this.model.get(Enums.ParameterRoles.X_MAJOR_TICK_HEIGHT) / 2,
                        yTh = this.model.get(Enums.ParameterRoles.Y_MAJOR_TICK_WIDTH) / 2,
                        xW = this.model.get(Enums.ParameterRoles.X_MAJOR_TICK_WIDTH) / 2,
                        yW = this.model.get(Enums.ParameterRoles.Y_MAJOR_TICK_HEIGHT) / 2;

                    return [
                        this.xAxis.getMajorTicks().map(function (tick) {
                            return [tick[0], tick[1] - xTh];
                        }),
                        this.yAxis.getMajorTicks().map(function (tick) {
                            return [tick[0] + yTh, tick[1] + yW];
                        })
                     ];
                },

                _getGridMinorTicks: function () {
                    var xTh = this.model.get(Enums.ParameterRoles.X_MINOR_TICK_HEIGHT) / 2,
                        yTh = this.model.get(Enums.ParameterRoles.Y_MINOR_TICK_WIDTH) / 2,
                        xW = this.model.get(Enums.ParameterRoles.X_MINOR_TICK_WIDTH) / 2,
                        yW = this.model.get(Enums.ParameterRoles.Y_MINOR_TICK_HEIGHT) / 2;


                    return [
                        this.xAxis.getMinorTicks().map(function (tick) {
                            return [tick[0] + xW, tick[1] - xTh];
                        }),
                        this.yAxis.getMinorTicks().map(function (tick) {
                            return [tick[0] + yTh, tick[1] + yW];
                        })
                    ];
                },

                getGridDimensions: function () {
                    return this.grid.getDimensions();
                },

                setActive: function (active) {
                    this.gridCanvas.active = active;
                    this.canvas.active = active;
                }

            });

            return BackLayer;
});