/** 
	* @class ConnectionHandler
	* @classdesc Represents an connectionHandler object. This object creates and maintains connections.
	
	*/
define(['common/Enums', 'core/connection/Connection', 'common/Error', 'common/Utilites', 'core/connection/ProjectConnections'], function (Enums, Connection, Error, Utilites, ProjectConnections) {
    /** 
	* Creates an ConnectionHandler object
	* @constructor
	* @this {ConnectionHandler}
	* @param {BaseAdapter} source Source of connection(context object).
	*/
    function ConnectionHandler() {
        this.connections = {};
        this.init();        
    };

    /** 
	* Adds the connection source and targets to connectionHandler.
	* @function add	
	* @this {ConnectionHandler}
	* @param {object{source, target, operation, convertation}} connectionSettings.
	*/
    ConnectionHandler.prototype.add = function (connectionSettings, ctx) {
        var connection = new Connection(connectionSettings, ctx);
        var fullPath = Utilites.buildFullPath(connectionSettings.source);

        if (!this.connections[fullPath]) {
            this.connections[fullPath] = [];
        }

        this.connections[fullPath].push(connection);

        return connection;
    };

    ConnectionHandler.prototype.updateOneTimeConnections = function (sender) {
        for (var propertyName in this.connections) {
            for (var i in this.connections[propertyName]) {
                if (this.connections[propertyName][i].isOneTime()) {
                    this.connections[propertyName][i].update(sender);
                }
            }
        }

        this.removeOneTimeConnections();
    },

    ConnectionHandler.prototype.removeOneTimeConnections = function (propertyName) {
        for (var propertyName in this.connections) {
            for (var i in this.connections[propertyName]) {
                if (this.connections[propertyName][i].isOneTime()) {
                    this.connections[propertyName][i].destroy();
                    this.connections[propertyName].splice(i, 1);
                }
            }
        }
    };

    ConnectionHandler.prototype.onchange = function (propertyName, sender, sourceOldValue) {
        var bracketRegx = /\[\d+\]/g,
            bracketExecResult,
            nextStop,
            i,
            connIndex,
            subPropertyName,
                targetConnections;

        var propertyPath = propertyName.substring(propertyName.indexOf('/') + 1);
        var bracketIndex = propertyName.search(this.bracketRegx);
        if (!this.connections[propertyName] && (bracketExecResult = bracketRegx.exec(propertyName)) != null) {
            //     ,    
            // -     ,   
            var bracketEntries = [];
            do{
                bracketEntries.push(bracketExecResult.index);
                bracketRegx.lastIndex = bracketExecResult.index + 1;
            }   while((bracketExecResult = bracketRegx.exec(propertyName)) != null);
            
            //       
            bracketEntries = bracketEntries.reverse();
            for(i=0;i<bracketEntries.length; i++){
                subPropertyName = propertyName.substr(0, bracketEntries[i]);
                targetConnections = this.connections[subPropertyName];
                if (targetConnections) {
                    //    -  
                    //.   ,      
                    //   this.connections[propertyName]  
                    // propertyName,  ,  source.propertyPath                  
                   
                    for (connIndex = 0; connIndex < targetConnections.length; connIndex++) {
                        var settings = $.extend(true, {}, targetConnections[connIndex].settings);
                        settings.source.propertyPath = propertyPath;
                        //propertyPath -  -,    ,    
                        settings.target.propertyPath = settings.target.propertyPath + propertyName.substr(subPropertyName.length, propertyName.length - subPropertyName.length)
                        this.add(settings, sender);
                    }
                }
            }      
        }

        for (var i in this.connections[propertyName]) {
            try {
                if (!this.connections[propertyName][i].isOneTime()) {
                    this.findFeedback(this.connections[propertyName][i]);
                    this.connections[propertyName][i].update(sender, sourceOldValue);
                }
            } catch (e) {
                Error.exception(e);
            }
        }       
    };

    ConnectionHandler.prototype.overrideOperation = function (propertyName, operation) {
        if (operation) {
            for (var i in this.connections[propertyName]) {
                this.connections[propertyName][i].settings.storedOperation = this.connections[propertyName][i].settings.operation;
                this.connections[propertyName][i].settings.operation = operation;
            }
        }
    };

    ConnectionHandler.prototype.restoreOperation = function (propertyName) {
        for (var i in this.connections[propertyName]) {
            if (this.connections[propertyName][i].settings.storedOperation) {
                this.connections[propertyName][i].settings.operation = this.connections[propertyName][i].settings.storedOperation;
                this.connections[propertyName][i].settings.storedOperation = undefined;
            }
        }
    };

    ConnectionHandler.prototype.updateAll = function (sender) {
        for (var group in this.connections) {
            for (var i in this.connections[group]) {
                //    
                //        
                //     -      
                if (this.connections[group][i].source.settings.type != Enums.connectionType.action &&
                    this.connections[group][i].source.settings.type != Enums.connectionType.event &&
                    !this.connections[group][i].isOneTime()) {
                    this.findFeedback(this.connections[group][i]);
                    this.connections[group][i].update(sender);
                }
            }           
        }
    };

    ConnectionHandler.prototype.getConnections = function (sourcePathRegex, targetPathRegex, type) {
        var searchBySource = (sourcePathRegex !== undefined) && (type === Enums.connectionSearchType.source || type === Enums.connectionSearchType.both);
        var searchByTarget = (targetPathRegex !== undefined) && (type === Enums.connectionSearchType.target || type === Enums.connectionSearchType.both);
        var foundConnections = [];
        var foundConnectionPath = null;

        for (var connetionSourcePath in this.connections) {
            //    ,   
            if (searchBySource && !sourcePathRegex.test(connetionSourcePath)) {
                continue;
            }

            if (searchByTarget === false) {
                foundConnections.push.apply(foundConnections, this.connections[connetionSourcePath]);
                continue;
            }

            var targetPath;
            this.connections[connetionSourcePath].forEach(function (connection) {
                targetPath = Utilites.buildFullPath(connection.settings.target);
                if (targetPathRegex.test(targetPath)) {
                    foundConnections.push(connection);
                }
            }, this);
        }

        return foundConnections;
    };

    ConnectionHandler.prototype.removeConnections = function (connsToRemove) {
        var i;      

        connsToRemove.forEach(function (removeConnection) {
            for (var fullPath in this.connections) {
                _.remove(this.connections[fullPath], function (connection) {
                    return connection === removeConnection;
                }, this);

                if (this.connections[fullPath].length === 0) {
                    delete this.connections[fullPath];
                }
            }
        }, this);
    }

    ConnectionHandler.prototype.findFeedback = function (connection) {
        if ("backward" in connection) { //   ,    
            return;
        }
        for (var connetionSourcePath in this.connections) {
            for (var i = 0; i < this.connections[connetionSourcePath].length; i++) {
                if (this.isFeedback(connection, this.connections[connetionSourcePath][i])) {
                    connection.backward = this.connections[connetionSourcePath][i];
                    this.connections[connetionSourcePath][i].backward = connection;
                    return;
                }
            }
        }
        connection.backward = null; //   
    }
    ConnectionHandler.prototype.isFeedback = function (connection1, connection2) { //       
        return (connection1.settings.target.path == connection2.source.settings.path
                && connection1.settings.source.path == connection2.target.settings.path
                && connection1.settings.target.propertyPath == connection2.source.settings.propertyPath
                && connection1.settings.source.propertyPath == connection2.target.settings.propertyPath
                && connection1.settings.target.itemId == connection2.source.settings.itemId
                && connection1.settings.source.itemId == connection2.target.settings.itemId
                );
    },

    ConnectionHandler.prototype.resetLastValues = function () {
        for (var connetionSourcePath in this.connections) {
            for (var i = 0; i < this.connections[connetionSourcePath].length; i++) {
                if (this.connections[connetionSourcePath][i].backward) {
                    this.connections[connetionSourcePath][i].backward.lastValue = undefined;
                }
            }
        }
    },
    /** 
	* Initializes a connectionHandler object. Processes connections settings.
	* @function init	
	* @this {Connection}
	*/
    ConnectionHandler.prototype.init = function () {
    };

    return ConnectionHandler;
});