﻿// <copyright file="TaskWrapper.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>
//


define(['common/Enums', 'base/EventTarget', 'when', 'common/Error', 'core/functionBlocks/TaskTimer', 'core/functionBlocks/ST/TaskWrapperModel', 'core/functionBlocks/ST/SoundService', 'helpers/PermissionChecker'],
    function (Enums, EventTarget, when, Error, Timer, TaskWrapperModel, SoundService, PermissionChecker) {

        var TaskWrapper = Class.extend({
            TASK_PROPERTY_CHANGED: 'TaskPropertyChanged',
            TASK_EXECUTED: 'TaskExecuted',
            init: function (metadata) {
                this.eventTarget = new EventTarget();
                this.metadata = metadata || {};
                this.global={};
                this.global.username = PermissionChecker.getOperator();
                this.model = new TaskWrapperModel(this.metadata.fbMetadata);
                this.model.subscribePropertyChanged(this._onPropertyChanged, this);
                this._callInternalBind = this._callInternal.bind(this);
                this.changedParams = {}; // path: value
                this.trackChange = true;
                this.timer = new Timer(+this.metadata.interval);
                this.timer.append(this.execute.bind(this));
                this.timer.append(this.afterExecute.bind(this));
                this.explicitCall = null;
            },

            start: function () {
                this.timer.start();
            },

            call: function (fbName, inputParams, outputParamNames, callback) {
                if (!this.timer.promise) {
                    //если промиса нету - то вызов попал между тиками таймера
                    this.timer.promise = this._callInternalBind(fbName, inputParams, outputParamNames).then(callback);                    
                } else {
                    //иначе таймер уже работает - добавляем в очередь
                    this.explicitCall = function (context, fb, input, output, callback) {
                        return function () {
                            return context._callInternalBind(fb, input, output).then(callback);
                        }
                    }(this, fbName, inputParams, outputParamNames, callback);
                }

                //вызывающий then всегда будет после отработки явного вызова ФБ
                return this.timer.promise;
            },

            _callInternal: function (fbName, inputParams, outputParamNames) {
                for (paramName in inputParams) {
                    this.model.set(paramName, inputParams[paramName]);
                }

                return this.execute(fbName).then(function () {
                    var outParams = {};
                    outputParamNames.forEach(function (paramName) {
                        outParams[paramName] = this.model.get(paramName);
                    }, this);

                    return outParams;
                }.bind(this));
            },

            execute: function (fbName) {
                return this.executeInternal(fbName).then(this._updateModel.bind(this))
                    .then(this._taskExecuted.bind(this));
            },

            afterExecute: function () {
                this.timer.promise = null;

                if (this.explicitCall !== null) {
                    var exp = this.explicitCall;
                    this.explicitCall = null;
                    return exp();
                }

                return when.resolve();
            },

            executeInternal: function (fbName) {
                if (this.isSupportsWorker()) {
                    return this._executeInWorker(this.metadata.fileName, fbName);
                } else {
                    return this._executeInTimer(this.metadata.fileName, fbName);
                };
            },

            isSupportsWorker: function () {
                return window.Worker !== undefined;
            },

            _executeInWorker: function (fileName, fbName) {
                return when.promise(function (resolve, reject) {
                    this._initWorker(resolve, reject, fileName);
                    this.changedParams.global = { username: this.global.username };
                    var p = {
                        fbName: fbName,
                        input: this.changedParams
                    };
                    this.changedParams = {};
                    this.worker.postMessage(p);
                }.bind(this));
            },
            _initWorker: function (resolve, reject, fileName) {
                if (this.worker === undefined) {
                    this.worker = new Worker(fileName);
                }
                this.worker.onmessage = function (e) {
                    switch (e.data.type) {
                        case "sound":
                            SoundService.call(e.data.params);
                            break;
                        default:
                            resolve(e.data);
                    }
                };
                this.worker.onerror = function (e) {
                    reject(e);
                };
                //this.worker.postMessage({
                //    input: { global: { username: this.global.username } }
                //});
            },

            _updateModel: function (fbResult) { //path: value
                this.trackChange = false;

                _.forOwn(fbResult, function (value, path) {
                    this.model.setParam(path, value);
                }, this);

                this.trackChange = true;
            },

            _taskExecuted: function () {
                this.fireTaskExecuted();

            },

            _onPropertyChanged: function (event) {
                if (this.trackChange === true) {
                    this.changedParams[event.property] = this.model.getParam(event.property);
                }

                this.fireTaskPropertyChanged(event.property, event.newValue, event.oldValue);
            },

            _executeInTimer: function (fileName, fbName) {
                return when.promise(function (resolve, reject) {
                    if (this.script == undefined) {
                        this._loadScriptFile(fileName).then(function (fileContent) {
                            this.script = eval(fileContent);
                            this._executeLoadedScript();
                        }.bind(this)).catch(function (err) {
                            Error.onerror(err);
                            reject(err);
                        });
                    } else {
                        this._executeLoadedScript();
                    }
                }.bind(this));
            },

            _executeLoadedScript: function () {
                setTimeout(function () {
                    resolve(this.script.self.onmessage(this.metadata.task));
                }.bind(this), 0);
            },

            _loadScriptFile: function (fileName) {
                return when.promise(function (resolve, reject) {
                    curl(['js!' + fileName]).then(function (file) {
                        resolve(file);
                    },
                        function (err) {
                            reject(err);
                        });
                });
            },

            subscribeTaskPropertyChanged: function (handler, context) {
                this.eventTarget.addListener(this.TASK_PROPERTY_CHANGED, handler, context);
            },

            unsubscribeTaskPropertyChanged: function (handler) {
                this.eventTarget.removeListener(this.TASK_PROPERTY_CHANGED, handler);
            },

            fireTaskPropertyChanged: function (property, newValue, oldValue) {
                this.eventTarget.fire({
                    type: this.TASK_PROPERTY_CHANGED,
                    target: this,
                    property: property,
                    newValue: newValue,
                    oldValue: oldValue,
                    taskName: this.metadata.name
                });
            },

            fireTaskExecuted: function () {
                this.eventTarget.fire({
                    type: this.TASK_EXECUTED,
                    target: this
                });
            },

            subscribeSubscribeTaskExecutedOnce: function (handler, context) {
                this.eventTarget.bindOnce(this.TASK_EXECUTED, handler, context);
            },

            subscribeTaskExecuted: function (handler, context) {
                this.eventTarget.addListener(this.TASK_EXECUTED, handler, context);
            },

            unsubscribeTaskExecuted: function (handler) {
                this.eventTarget.removeListener(this.TASK_EXECUTED, handler);
            },


        });

        return TaskWrapper;
    });