﻿// <copyright file="HeartbeatManager.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>
// _completedRequest
define(['base/ObservableObject', 'server/adapters/DataAdapter', 'when', 'core/Timer', 'server/ServerStateManager',
    'common/Enums', 'core/global/GlobalEvents', 'core/global/GlobalSettingsType', 'server/ServerLogger', 'core/Statistic', 'core/L10n', 'common/Error'],
    function (ObservableObject, DataAdapter, when, Timer, ServerStateManager, Enums,
        GlobalEvents, GlobalSettings, ServerLogger, Statistic, l10n, Error) {
        var SERVER_CONNECTION_ERROR = 'ServerConnectionError';
        var EMPTY = '';
        var HeartbeatManager = ObservableObject.extend({

            init: function () {
                this._super();
                this.interval = 5000;
                this.className = 'HeartbeatManager';
                this.projectId = EMPTY;
                this.needReload = false;
                this.trackStatusBind = this.trackStatus.bind(this);
                this.servers = {};
                var ctx = this;
                this.currentIP = null;
                this.haveBackup = false;

                if (GlobalSettings.MainIP.value) {
                    GlobalSettings.MainIP.value.concat(GlobalSettings.RedundancyIP.value).forEach(function (el) {
                        if (el) {
                            ctx.servers["http://" + el + ":" + GlobalSettings.WebPort.value] = false;
                        }
                    });

                    if (!!(GlobalSettings.RedundancyIP &&
                        GlobalSettings.RedundancyIP.value &&
                        (GlobalSettings.RedundancyIP.value.length > 0))) {
                        if (this.servers[document.location.origin] != undefined) {
                            this.currentIP = document.location.origin;
                            this.haveBackup = true;
                        } else if (this.servers[document.location.origin + ":80"] != undefined) {
                            this.currentIP = document.location.origin + ":80";
                            this.haveBackup = true;
                        }
                        else
                            Error.warn("Current address is not found : " + document.location.origin);                        
                    }
                }

                if (this.haveBackup)
                    this.serversList = Object.keys(this.servers);
                else
                    this.serversList = [""];
            },

            initInternal: function () {
                this.initialized = false;
                this.serverStateManager = new ServerStateManager();
                this.dataAdapter = new DataAdapter(this.serverStateManager);
                this.timer = new Timer(this.interval);
                this.timer.append(this.trackStatusBind);

                this.trackStatus().then(function () {
                    this.startTimer();
                }.bind(this));
            },

            startTimer: function () {
                this.timer.start();
            },

            trackStatus: function() {
                var ctx = this;
                return when.promise(function(resolve) {
                    ServerLogger.sendStatistics(l10n.get('messages.statisticsSentMessage'),
                        Statistic.getStatisticReport());
                    ctx.serversList.forEach(function (prop) {
                        ctx.dataAdapter.getState(prop).then(function (responce) {
                                if (ctx.haveBackup) {
                                    if (ctx.currentIP != responce.data.url && responce.data.projectId && ctx.projectId && responce.data.projectId !== ctx.projectId) //может быть что не на все узлы загрузилась конфигурация, тогда его ответы нужно игнорировать
                                        return resolve();
                                    ctx.servers[responce.data.url] = responce.data.isMaster;
                                    if (ctx.currentIP == responce.data.url && !responce.data.isMaster ||
                                        ctx.currentIP != responce.data.url && responce.data.isMaster) {
                                        //текущий узел больше не master или master-ом стал другой узел
                                        ctx.currentIP = ctx.getMasterIP();
                                        if (ctx.currentIP)
                                            window.open(ctx.currentIP, '_self');
                                    } else {
                                        return resolve();
                                    }

                                    var isNotMain = GlobalSettings.MainIP.value.find(function (el) {
                                    return (responce.data.url.indexOf(el) != -1);
                                });
                                    if (responce.data.isMaster && !isNotMain && ctx.currentIP != responce.data.url
                                ) { //master-ом стал резервный, можно не ждать отрицательного ответа от основного
                                    window.open(responce.data.url, '_self');
                                }
                            }
                                if (!ctx.projectId && responce.data.projectId && !ctx.needReload) {
                                    ctx.projectId = responce.data.projectId;
                                    ctx.needReload = false;
                                } else if (responce.data.projectId && ctx.projectId && responce.data.projectId !== ctx.projectId) {
                                    ctx.onReloadProject();
                                }
                                ctx.fireServerErrorStateChanged({
                                    isError: false,
                                    id: Enums.constants.HEARTBEAT_MANAGER
                                });
                                resolve();
                            })
                        .catch(function(err) {
                            if (ctx.projectId === EMPTY) {
                          //если запрос провалился еще до получения id,
                          //то мы не можем доверять следующему - на сервер мог быть залит
                          //другой проект, а мы, приняв его id, оставим предыдущую страницу
                            ctx.needReload = true;
                        }
                            ctx.fireServerErrorStateChanged(
                            { isError: true, id: Enums.constants.HEARTBEAT_MANAGER });
                            resolve();
                        });
                    });
                });
            },

            stopTimer: function () {
                this.timer.stop();
            },

            getMasterIP: function () {
                for (var prop in this.servers) {
                    if (prop != this.currentIP && this.servers[prop] == true) {
                        return prop;
                    }
                }
                return null;
            },
            onReloadProject: function () {
                window.location.reload(true);
            },

            subscribeServerErrorStateChanged: function (handler, context) {
                this.eventTarget.addListener(SERVER_CONNECTION_ERROR, handler, context);
            },

            unsubscribeServerErrorStateChanged: function (handler) {
                this.eventTarget.removeListener(SERVER_CONNECTION_ERROR, handler);
            },

            fireServerErrorStateChanged: function (event) {
                this.eventTarget.fire({
                    type: SERVER_CONNECTION_ERROR,
                    target: this,
                    isError: event.isError,
                    vmId: event.id
                });
            }
        });

        return HeartbeatManager;
    });