// <copyright file="VariablesManager.js" company="">
// , 2014
// </copyright>
// 


/** 
	* @class VariablesManager
	* @classdesc Represents an object, which holds and manages state of "server variable" abstraction.
	
	*/
define(['base/ObservableObject', 'server/adapters/DataAdapter', 'when', 'core/Timer', 'server/ServerStateManager', 'common/Enums',
'common/Utilites'],
    function (ObservableObject, DataAdapter, when, Timer, ServerStateManager, enums, Utilities) {
        var SERVER_CONNECTION_ERROR = 'ServerConnectionError';

        var VariablesManager = ObservableObject.extend({
            stateVars: enums.serverState,
            keyFieldName: 'itemId',
            pathFieldName: 'path',
            valueFieldName: 'value',
            sendTimestampFieldName: 'sendTimestamp',
            maxEntitySize: 200,
            SERVER_DATA_READY: 'ServerDataReady',
        
            init: function (options) {               
                this.options = options;
                this._super();
                this.sendGlobalCounter = 0;
                this.initInternal();
                this.className = 'VariablesManager';
            },
            initInternal: function () {
                this.clientHandle = 1;
                this.initialized = false;
                this.variables = {};
                this.variables = {
                    get: {
                        protocol: [],
                        schema: {}, //  
                        callback: {} // clientHandle
                    },
                    set: {
                        protocol: [], //to set
                        schema: {} //all
                    }
                };

                // VariablesManager   .   
                //         
                //      
                this.lastSubscribeVars = [];              
                this.serverStateManager = new ServerStateManager();
                this.serverStateManager.subscribePropertyChanged(this._onServerErrorStateChanged, this);
                this.serverStateManager.set(this.stateVars.requestedPublishingInterval,
                    this.options.parameters.Interval.value);
                this.dataAdapter = this._createDataAdapter(this.options);
                this.sGet = this.serverStateManager.get.bind(this.serverStateManager);
                this.getAllWrap = this.getAll.bind(this);
                this._initializeTimerOnCreate();
            },

            setOptions: function (options) {
                this.options = options;
                if (options.interval) {
                    this.serverStateManager.set(this.stateVars.requestedPublishingInterval, options.interval);
                }
            },

            _createDataAdapter: function (options) {
                return new DataAdapter(this.serverStateManager);
            },

            _initializeTimerOnCreate: function () {
                this.initialize();
            },

            initialize: function () {
                this.timer = new Timer(this.sGet(this.stateVars.requestedPublishingInterval));
                this.timer.append(this.setAll.bind(this));
                // Vlad:  .  ,      .
                //     .
                //this.startTimer();
            },

            startTimer: function () {
                this.timer.interval = this.sGet(this.stateVars.requestedPublishingInterval);
                this.timer.start();
            },

            stopTimer: function () {
                this.timer.stop();
            },
            getKey: function (variable){
                return variable[this.keyFieldName]+variable[this.pathFieldName];
            },

            addToSubscription: function (vars) {
                var i;
                if (vars.getValues !== undefined) {
                for (i = 0; i < vars.getValues.length; i++) {
                    var variable = vars.getValues[i];
                    //   -   
                    if (this.variables.get.schema[this.getKey(variable)] === undefined) {
                        this._addGetVariableToSubscription(variable);
                    } else {                       
                        //     
                        var existGetVariable = this.variables.get.schema[this.getKey(variable)];
                        existGetVariable.usesCounts++;
                    }
                }
                }

                if (vars.setValues !== undefined) {
                for (i = 0; i < vars.setValues.length; i++) {
                    var variable = vars.setValues[i];
                    var existSetVariable = this.variables.set.schema[this.getKey(variable)];
                    if (existSetVariable === undefined) {
                        this.variables.set.schema[this.getKey(variable)] = variable;
                        variable.usesCounts = 1;
                    } else {
                        existSetVariable.usesCounts++;
                    }
                }
                }
            },

            _addGetVariableToSubscription: function (variable) {
                variable.clientHandle = this.clientHandle++;
                variable.usesCounts = 1;
                this.variables.get.protocol.push(variable);
                this.lastSubscribeVars.push(variable);

                this.variables.get.schema[this.getKey(variable)] = variable;
                this.variables.get.callback[variable.clientHandle] = variable;
            },

            subscribe: function () {
                var def = when.defer();
               //console.log('Subscribing ');
                var ctx = this;

                //    
                if (this.lastSubscribeVars.length > 0) {
                    ctx.dataAdapter.initialize().then(function () {
                        //console.log('Adding monitoring items');
                        if (ctx.lastSubscribeVars.length > 0) {
                            ctx._addMonitoredData({ data: { items: [] } }).then(function (result) {
                                if (!result.data.items) {
                                    def.reject('CreateMonitoredDataItems  ,   : ' + result.data);
                                }
                                var items = result.data.items;
                                //console.log('Monitoring items added: ');
                                //console.log(items);
                                var deletedMonitoredItemsIds = [];
                                for (var i = 0; i < items.length; i++) {
                                    //      
                                    if (ctx.variables.get.callback[items[i].clientHandle]) {
                                        ctx.variables.get.callback[items[i].clientHandle].monitoredItemId = items[i].monitoredItemId;
                                    } else {
                                        //   
                                        deletedMonitoredItemsIds.push(items[i].monitoredItemId);
                                    }
                                }

                                if (deletedMonitoredItemsIds.length > 0) {
                                    ctx._unsubscribeVars(deletedMonitoredItemsIds);
                                }

                                ctx.lastSubscribeVars = [];
                                //console.log('Refreshing...');
                                ctx.dataAdapter.refresh().then(function (result) {
                                    ctx._addGetTimer();
                                    def.resolve();
                                });
                            });
                        }
                    });
                } else {
                    def.resolve();
                }

                return def.promise;
            },

            _addGetTimer: function () {
                //console.log('Refreshed');
                //      -     
                if (this.initialized === false) {
                    //console.log('Init timer');
                    this.initialized = true;
                    // Vlad:   getValues  .
                    this.timer.setInterval(this.sGet(this.stateVars.requestedPublishingInterval));
                    this.timer.prepend(this.getAllWrap);
                }
            },

            _addMonitoredData: function (res) {
                var ctx = this;
                ctx.res = res;
                if (this.lastSubscribeVars.length > 0) {
                    return this.dataAdapter.addMonitoredData(this.lastSubscribeVars.splice(0, this.maxEntitySize)).then(function (result) {
                        ctx.res.data.items = ctx.res.data.items.concat(result.data.items);
                        return ctx._addMonitoredData(ctx.res);
                    });
                }
                return ctx.res;
            },

            unsubscribe: function (vars) {
                var def = when.defer();
                var i;
                //console.log('Unsubscribing :');
                //console.log('Unsubscribing set');
                //set -  
                if (vars.setValues !== undefined) {
                for (i = 0; i < vars.setValues.length; i++) {
                    var variable = vars.setValues[i];
                    var existSetVariable = this.variables.set.schema[this.getKey(variable)];
                    if (existSetVariable && existSetVariable.usesCounts === 1) {
                        //console.log('Remove varibale');
                        //console.log(existSetVariable);
                        delete this.variables.set.schema[this.getKey(variable)];
                        var index = _.findIndex(this.variables.set.protocol, function (protocolVariable) {
                            return protocolVariable[this.keyFieldName] == variable[this.keyFieldName];
                        }, this);                    
                    } else {
                        //console.log('Decrease counter');
                        existSetVariable.usesCounts--;
                        //console.log(existSetVariable);
                    }
                }
                }

                //get -  + 
                var monitoredItemsIds = [];
                if (vars.getValues !== undefined) {
                for (i = 0; i < vars.getValues.length; i++) {                    
                    var variable = vars.getValues[i];
                    var removedVariable = this.variables.get.schema[this.getKey(variable)];
                    if (removedVariable.usesCounts === 1) {
                        //console.log('Remove varibale');
                        //console.log(removedVariable);
                        //.     ,   addToSubscription
                        //  ,          
                        //       
                        removedVariable.value = undefined;
                        // undefined -    ,
                        //       CreateMonitoredDataItems
                        if (removedVariable.monitoredItemId !== undefined) {
                            monitoredItemsIds.push(removedVariable.monitoredItemId);
                        }

                        this._removeVariable(variable, removedVariable);

                        } else {
                        //console.log('Decrease counter');
                        removedVariable.usesCounts--;
                        //console.log(removedVariable);
                    }
                }
                }

                if (monitoredItemsIds.length > 0) {                  
                    //       -  
                    if (this._canDeleteSubscription() === true) {
                        //console.log('Delete subscription');
                        this.initialized = false;
                        //  , . -   
                        this.timer.remove(this.getAllWrap);                        
                        this._deleteSubscription().then(function () {
                            def.resolve();
                        });
                    } else {                       
                        this._unsubscribeVars(monitoredItemsIds).then(function () {
                            def.resolve();
                        }, function () {
                            def.reject('Unsubscribe Vars rejected');
                        });
                    }
                }
                else {
                    def.resolve();
                }

                return def.promise;
            },

            _deleteSubscription: function () {
                return this.dataAdapter.destroy();
            },

            _unsubscribeVars: function (monitoredItemsIds) {
                return this.dataAdapter.deleteMonitoredEvent(monitoredItemsIds);
            },

            _removeVariable: function (variable, removedVariable) {
                var index = _.findIndex(this.variables.get.protocol, function (protocolVariable) {
                    return (this.getKey(protocolVariable) == this.getKey(variable));
                }, this);

                if (index != -1) {
                    this.variables.get.protocol.splice(index, 1);
                } else {
                    console.warn('Variable not found in get.protocol' + this.getKey(variable));
                }

                delete this.variables.get.schema[this.getKey(variable)];
                delete this.variables.get.callback[removedVariable.clientHandle];
            },

            _canDeleteSubscription: function () {
                return this.variables.get.protocol.length === 0;
            },

            getValue: function (id, property) {
                console.assert(this.variables.get.schema[id][property] !== undefined);
                return this._getProperty(id, true)[property];                   
            },

            getProperty: function (id, propertyPath) {
                return this._getProperty(id+propertyPath, true);
            },

            getPropertyType: function (id, propertyPath) {
                var property = this._getProperty(id+propertyPath, false);
                if (property === undefined) {
                    console.warn('Possible error: cannot get type of undefinied: ' + id);
                    return undefined;
                }

                return property.typeName;
            },

            _getProperty: function (id, isCopy) {                
                if (this.variables.get.schema[id] !== undefined) {
                    return this._getPropertyValue(this.variables.get.schema[id], isCopy);
                } else if (this.variables.set.schema[id] !== undefined) {
                    return this._getPropertyValue(this.variables.set.schema[id], isCopy);
                }
                console.warn('GetValue: Requested variable' + id + ' not found!!!!');
            },

            _getPropertyValue: function (property, isCopy) {
                if (isCopy === false) {
                    return {
                        typeName: property.type,
                        value: property.value
                    }
                } else {
                    var copy = $.extend(true, {}, property);
                    copy.typeName = copy.type;
                    delete copy.type;
                    return copy;
                }
            },

            setProperty: function (id, propertyPath, property, operation) {
                var oldProperty = this._getProperty(id+propertyPath, false);               
                this.setValue(id, propertyPath, property.value, operation);
            },

            setValue: function (id, property, value, operation) {
                console.assert(value !== undefined);
                //  
                var variable = this.variables.set.schema[id+property];
                variable[this.valueFieldName] = value;
                variable[this.sendTimestampFieldName] = this.sendGlobalCounter++;
                
                //         
                if (this.variables.set.protocol.indexOf(variable) == -1) {
                    this.variables.set.protocol.push(variable);
                    variable.operation = operation || "move";
                }
            },

            getAll: function () {
                var that = this;
                if (this.initialized && this.serverStateManager.get(this.stateVars.Connected)) {
                    return this.dataAdapter.getAll().then(function (recs) {
                            return that._getAllSucceed(recs);
                        }).catch(function (result) {                       
                            return that._getAllFailed(result);
                        });
                }

                return when.resolve();
            },

            _getAllSucceed: function (recs) {                
                var i;
                for (i = 0; i < recs.length; i++) {
                    //     ,    
                    var variable = this.variables.get.callback[recs[i].clientHandle];
                    if (variable !== undefined) {
                        this._updateVariable(variable, recs[i]);
                    } else {
                        //console.log('Fire update problem: itemId:' + variable[this.keyFieldName] + ' old:' + oldvalue + ' new:' + variable.value);
                        console.warn('GetAllSucceed: Requested variable not found!!!!');
                    }
                }                

                this._afterGetAllSucceed();

                return when.resolve();
            },

            _updateVariable: function (variable, record) {
                variable.statusCode = record.statusCode !== undefined ? record.statusCode : 0;
                if (Utilities.compare(variable.value, record.value) == 0) {
                    return;
                }

                var oldvalue = variable.value;
                variable.value = record.value;
                //console.log('Fire update: itemId:' + variable[this.keyFieldName] + ' old:' + oldvalue + ' new:' + variable.value);                
                this.firePropertyChanged(this._getVariablePropertyPath(variable), oldvalue, variable.value);
            },

            _serverDataReady: function () {
                //       ,
                //    
                this.fireServerDataReady();
                this.eventTarget.removeAllListeners(this.SERVER_DATA_READY);
            },

            _afterGetAllSucceed: function () {
                this._serverDataReady();
            },

            _getAllFailed: function () {
                return when.reject(new Error("getAll failed"));
            },

            setAll: function () {  
                var def = when.defer();
                var ctx = this;
                if (this.variables.set.protocol.length > 0) {
                    this.writeDataTimeSpan = this.sendGlobalCounter;
                    this.dataAdapter.writeData(this.variables.set.protocol).then(function () {
                        _.remove(ctx.variables.set.protocol, function (variable) {
                            return variable[this.sendTimestampFieldName] < this.writeDataTimeSpan;
                        }, ctx);
                        def.resolve();
                    });
                }
                else {
                    def.resolve();
                }

                return def.promise;
            },

            _onServerErrorStateChanged: function (event) {
                switch (event.property) {
                    case this.stateVars.connectionError:
                        this.fireServerErrorStateChanged({
                            isError: this.serverStateManager.get(this.stateVars.connectionError),
                            id: this.options.id
                        });
                        break;
                    case this.stateVars.wasRecconected:
                        if (this.serverStateManager.get(this.stateVars.wasRecconected) == true) {
                            this.serverStateManager.set(this.stateVars.wasRecconected, false);                          
                        }
                        break;
                    case this.stateVars.Connected:
                        if (!this.sGet(this.stateVars.Connected)) {
                            this.lastSubscribeVars = this.variables.get.protocol;
                            this.serverStateManager.set(this.stateVars.SubscriptionId, -1);
                            this.subscribe();
                        }                        
                        break;
                }
            },

            _getVariablePropertyPath: function (variable) {
                return variable[this.keyFieldName] + '/' + variable[this.pathFieldName];
            },

            subscribeServerDataReady: function (handler, context) {
                this.eventTarget.addListener(this.SERVER_DATA_READY, handler, context);
            },

            subscribeServerDataReadyOnce: function (handler, context) {
                this.eventTarget.bindOnce(this.SERVER_DATA_READY, handler, context);
            },

            fireServerDataReady: function (windowId) {
                this.eventTarget.fire({
                    type: this.SERVER_DATA_READY,
                    target: this,
                    windowId: windowId,
                    sourceId: this.options.id
                });
            },            

            unsubscribeServerDataReady: function (handler) {
                this.eventTarget.removeListener(this.SERVER_DATA_READY, handler);
            },

            subscribeServerErrorStateChanged: function (handler, context) {
                this.eventTarget.addListener(SERVER_CONNECTION_ERROR, handler, context);
            },

            unsubscribeServerErrorStateChanged: function (handler) {
                this.eventTarget.removeListener(SERVER_CONNECTION_ERROR, handler);
            },

            fireServerErrorStateChanged: function (event) {
                this.eventTarget.fire({
                    type: SERVER_CONNECTION_ERROR,
                    target: this,
                    isError: event.isError,
                    vmId: event.id
                });
            }
            
        });

        return VariablesManager;
    });