﻿// <copyright file="GridPanelController.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>
// 


/*
* @class GridPanelController controller class for GridPanel control
*/
define(['common/Enums', 'base/ContainerController'], function (Enums, ContainerController) {

    var GridPanelController = ContainerController.extend({

        cells: null,
        rowCells: null,
        colCells: null,

        isInitialized: false,

        init: function () {
            this._super();

            this.ClassName = 'GridPanelController';

            this.cells = [];
            this.rowCells = [];
            this.colCells = [];
        },

        onChildrenAdded: function (event) {
            this._super(event);

            initEmptyItems(this.rowCells, this.mvc.model.getRowDefinition().length);
            initEmptyItems(this.colCells, this.mvc.model.getColumnDefinition().length);

            this.isInitialized = true;          
        },

        onAddedToDOM: function () {
            this._super();
            this.fullReposition();
        },

        fullReposition: function () {
            this.repositionCells();
            this.realignControls();
        },

        appendChild: function (childControl) {

            var cellEl = document.createElement('div');
            var cell = $(cellEl),
                cellId = this.buildCellId(childControl);

            cell.addClass(cellId);
            cell.addClass('grid-cell');

            this.cells.push(cell);

            childControl.model.set(Enums.ParameterRoles.WIDTH_UNITS, Enums.SizeType.Pixel);
            childControl.model.set(Enums.ParameterRoles.HEIGHT_UNITS, Enums.SizeType.Pixel);

            this.addCellToRow(cell, childControl);
            this.addCellToCol(cell, childControl);

            this.mvc.view.$().append(cell);

            this.mvc.append(childControl, '.' + cellId);
        },

        buildCellId: function (childControl) {
            //var id = childControl.model.get(Enums.ParameterRoles.ROW) + '_' +
            //    childControl.model.get(Enums.ParameterRoles.COL) + '_' +
            //    childControl.model.get(Enums.ParameterRoles.ROWPAN) + '_'
            //    childControl.model.get(Enums.ParameterRoles.COLSPAN);

            return 'c' + childControl.model.getId();
        },

        addCellToRow: function (cell, control) {
                    
            var rowNumber = control.model.get(Enums.ParameterRoles.ROW) || 0;

            if (!this.rowCells[rowNumber]) {
                this.rowCells[rowNumber] = [];
            }

            this.rowCells[rowNumber].push({ cell: cell, control: control });
        },

        addCellToCol: function (cell, control) {

            var colNumber = control.model.get(Enums.ParameterRoles.COL) || 0;

            if (!this.colCells[colNumber]) {
                this.colCells[colNumber] = [];
            }

            this.colCells[colNumber].push({ cell: cell, control: control });
        },

        modelPropertyChangedInternal: function (event) {

            this._super(event);

            var value = this.mvc.model.get(event.property);

            switch (event.property) {
                case Enums.ParameterRoles.COLUMN_DEFINITION:
                    this.onColumnDefinitionChanged(value);
                    break;
                case Enums.ParameterRoles.ROW_DEFINITION:
                    this.onRowDefinitionChanged(value);
                    break;
            }
        },

        onColumnDefinitionChanged: function (value) {
            //do nothing because we do not support dynamization
        },

        onRowDefinitionChanged: function (value) {
            //do nothing because we do not support dynamization
        },

        onBorderThiknessChanged: function (value) {
            this._super();

            this.repositionCells();
        },

        childModelPropertyChangedInternal: function (event) {
            this._super();

            switch (event.property) {
                case Enums.ParameterRoles.COL:
                    this.onChildColChanged(event);
                    break;
                case Enums.ParameterRoles.ROW:
                    this.onChildRowChanged(event);
                    break;
                case Enums.ParameterRoles.COLPAN:
                    this.onChildColspanChanged(event);
                    break;
                case Enums.ParameterRoles.ROWSPAN:
                    this.onChildRowspanChanged(event);
                    break;
                case Enums.ParameterRoles.HORIZONTAL_ALIGN:
                    this.onChildHorizontalAlign(event);
                    break;
                case Enums.ParameterRoles.VERTICAL_ALIGN:
                    this.onChildVerticalAlign(event);
                    break;
                case Enums.ParameterRoles.WIDTH:
                case Enums.ParameterRoles.WIDTH_UNITS:
                    this.onChildWidthChanged(event);
                    break;
                case Enums.ParameterRoles.HEIGHT:
                case Enums.ParameterRoles.HEIGHT_UNITS:
                    this.onChildHeightChanged(event);
                    break;
            }
        },

        childModelPropertyChanging: function (event) {
            switch (event.property) {
                case Enums.ParameterRoles.WIDTH:
                case Enums.ParameterRoles.WIDTH_UNITS:
                    this.onChildChangingWidth(event);
                    break;
                case Enums.ParameterRoles.HEIGHT:
                case Enums.ParameterRoles.HEIGHT_UNITS:
                    this.onChildChangingHeight(event);
                    break;
            }
        },

        onChildChangingWidth: function (event) {
            var childAlign = event.target.get(Enums.ParameterRoles.HORIZONTAL_ALIGN);

            if (childAlign === Enums.HorizontalElementsAlign.Stretch) {
                event.silent = true;
                return;
            }
        },

        onChildChangingHeight: function (event) {
            var childAlign = event.target.get(Enums.ParameterRoles.VERTICAL_ALIGN);

            if (childAlign === Enums.VerticalElementsAlign.Stretch) {
                event.silent = true;
                return;
            }
        },

        onChildColChanged: function (event) {
            //do nothing because we do not support dynamization
        },

        onChildRowChanged: function (event) {
            //do nothing because we do not support dynamization
        },

        onChildColspanChanged: function (event) {
            //do nothing because we do not support dynamization
        },

        onChildRowspanChanged: function (event) {
            //do nothing because we do not support dynamization
        },

        onChildHorizontalAlign: function (event) {

            if (this.cells.length === 0) {
                //we are not initialized yet
                return;
            }

            var child = this.childById(event.target.getId());
            var cell = this.cells[this.childIndexById(event.target.getId())];           

            if (event.oldValue === Enums.HorizontalElementsAlign.Stretch) {
                child.controller.applyWidth();
            }

            this.applyChildHorizontalAlign(cell, child, event.newValue);
        },

        onChildVerticalAlign: function (event) {
            if (this.cells.length === 0) {
                //we are not initialized yet
                return;
            }

            var child = this.childById(event.target.getId());
            var cell = this.cells[this.childIndexById(event.target.getId())];

            if (event.oldValue === Enums.HorizontalElementsAlign.Stretch) {
                child.controller.applyHeight();
            }

            this.applyChildVerticalAlign(cell, child, event.newValue);
        },

        onChildWidthChanged: function (event) {
            //if child in col with auto size then reposition horizontally
            //otherwise reposition only control in cell

            if (this.cells.length === 0) {
                //we are not initialized yet
                return;
            }

            var childIndex;
            var colIndex = event.target.get(Enums.ParameterRoles.COL);
            var colDefinitions = this.mvc.model.getColumnDefinition();

            if (colDefinitions[colIndex] === 'auto') {
                this.repositionCellsHorizontally();
            }
            else if (event.target.get(Enums.ParameterRoles.HORIZONTAL_ALIGN) === Enums.HorizontalElementsAlign.Center) {

                childIndex = this.childIndexById(event.target.getId());

                if (!this.cells[childIndex]) {
                    //we are not initialized yet
                    return;
                }

                this.applyChildHorizontalAlign(this.cells[childIndex],
                                            this.orderedControls[childIndex]);
            }
        },

        onChildHeightChanged: function (event) {
            //if child in row with auto size then reposition vertically
            //otherwise reposition only control in cell

            if (this.cells.length === 0) {
                //we are not initialized yet
                return;
            }

            var childIndex;
            var rowIndex = event.target.get(Enums.ParameterRoles.COL);
            var rowDefinitions = this.mvc.model.getRowDefinition();

            if (rowDefinitions[rowIndex] === 'auto') {
                this.repositionCellsVertically();
            }
            if (event.target.get(Enums.ParameterRoles.VERTICAL_ALIGN) === Enums.VerticalElementsAlign.Center) {

                childIndex = this.childIndexById(event.target.getId());

                if (!this.cells[childIndex]) {
                    //we are not initialized yet
                    return;
                }

                this.applyChildVerticalAlign(this.cells[childIndex],
                                            this.orderedControls[childIndex]);
            }
        },

        fireActualWidthChanged: function (newValue) {
            this._super(newValue);
            this.repositionCellsHorizontally();
        },

        fireActualHeightChanged: function (newValue) {
            this._super(newValue);
            this.repositionCellsVertically();
        },

        repositionCells: function () {
            this.repositionCellsHorizontally();
            this.repositionCellsVertically();
        },

        realignControls: function () {
            var i;

            for (i = 0; i < this.orderedControls.length; i++) {
                this.applyChildHorizontalAlign(this.cells[i], this.orderedControls[i]);
                this.applyChildVerticalAlign(this.cells[i], this.orderedControls[i]);
            }
        },

        repositionCellsHorizontally: function () {
            var cols = this.getActualColWidths();
            var colIndex,
                colCells,
                cellIndex,
                colSpan,
                cell,
                cellWidth,
                clientRect = this.calcClientRect();

            if (this.colCells.length === 0) {
                return;
            }

            for (colIndex = 0; colIndex < cols.length; colIndex++) {

                colCells = this.colCells[colIndex];

                for (cellIndex = 0; cellIndex < colCells.length; cellIndex++) {
                    cell = colCells[cellIndex];

                    colSpan = cell.control.model.get(Enums.ParameterRoles.COLSPAN) || 1;
                    cellWidth = getCellSideSize(colIndex, colSpan, cols);

                    cell.cell.css({ left: clientRect.left + 'px', width: cellWidth + 'px' });

                    if (cell.control.model.get(Enums.ParameterRoles.HORIZONTAL_ALIGN) === Enums.HorizontalElementsAlign.Center) {
                        this.applyChildHorizontalAlign(cell.cell, cell.control);
                    }
                }

                clientRect.left += cols[colIndex];
            }
        },

        applyChildHorizontalAlign: function (cell, child, horizontalAlign) {
            if (!cell) {
                return;
            }

            if (!horizontalAlign) {
                horizontalAlign = child.model.get(Enums.ParameterRoles.HORIZONTAL_ALIGN);
            }
            switch (horizontalAlign) {
                case Enums.HorizontalElementsAlign.Left:
                    child.controller.setLeftAnchor('0px');
                    break;
                case Enums.HorizontalElementsAlign.Center:
                    var left = (cell.width() - child.controller.getControlWidthPx()) / 2;
                    child.controller.setLeftAnchor(Math.floor(left) + 'px');
                    break;
                case Enums.HorizontalElementsAlign.Right:
                    child.controller.setRightAnchor('0px');
                    break;
                case Enums.HorizontalElementsAlign.Stretch:
                    child.controller.setLeftAnchor('0px');
                    child.controller.setControlWidth('100%');
                    break;
            }
        },

        repositionCellsVertically: function () {
            var rows = this.getActualRowHeights();
            var rowIndex,
                rowCells,
                cellIndex,
                rowSpan,
                cell,
                cellHeight,
                clientRect = this.calcClientRect();

            if (this.rowCells.length === 0) {
                return;
            }

            for (rowIndex = 0; rowIndex < rows.length; rowIndex++) {

                rowCells = this.rowCells[rowIndex];

                for (cellIndex = 0; cellIndex < rowCells.length; cellIndex++) {
                    cell = rowCells[cellIndex];

                    rowSpan = cell.control.model.get(Enums.ParameterRoles.ROWSPAN);
                    cellHeight = getCellSideSize(rowIndex, rowSpan, rows);

                    cell.cell.css({ top: clientRect.top + 'px', height: cellHeight + 'px' });

                    if (cell.control.model.get(Enums.ParameterRoles.VERTICAL_ALIGN) === Enums.VerticalElementsAlign.Center) {
                        this.applyChildVerticalAlign(cell.cell, cell.control);
                    }
                }

                clientRect.top += rows[rowIndex];
            }
        },

        applyChildVerticalAlign: function (cell, child, verticalAlign) {
            if (!cell) {
                return;
            }

            if (!verticalAlign) {
                verticalAlign = child.model.get(Enums.ParameterRoles.VERTICAL_ALIGN);
            }
            switch (verticalAlign) {
                case Enums.VerticalElementsAlign.Top:
                    child.controller.setTopAnchor('0px');
                    break;
                case Enums.VerticalElementsAlign.Center:
                    var top = (cell.height() - child.controller.getControlHeightPx()) / 2;
                    child.controller.setTopAnchor(Math.floor(top) + 'px');
                    break;
                case Enums.VerticalElementsAlign.Bottom:
                    child.controller.setBottomAnchor('0px');
                    break;
                case Enums.VerticalElementsAlign.Stretch:
                    child.controller.setTopAnchor('0px');
                    child.controller.setControlHeight('100%');
                    break;
            }
        },

        getActualRowHeights: function () {
            var rowDefinitions = this.mvc.model.getRowDefinition();
            var clientRect = this.calcClientRect();
            var fixedHeight = 0,
                notFixedHeight = 0,
                proportionalPartsCount = 0,
                proportionalPartSize = 0;
            var actualRowDefinitions = rowDefinitions.slice(0);

            //1. look for fixed size rows
            //2. look for auto size rows
            //3. look for proportional size rows
            var fixedIndexes = [];//getFixedSizeItems(rowDefinitions);
            var autoIndexes = [];//getAutoSizeItems(rowDefinitions);
            var proportionalIndexes = [];//getProportionalItems(rowDefinitions);

            filterGridItems(rowDefinitions, fixedIndexes, autoIndexes, proportionalIndexes)

            if (fixedIndexes.length > 0) {
                fixedHeight = sumItems(rowDefinitions, fixedIndexes);
            }

            if (autoIndexes.length > 0) {
                fixedHeight += this.calcAutoRowsHeights(actualRowDefinitions, autoIndexes);
            }

            notFixedHeight = clientRect.height - fixedHeight;
            if (proportionalIndexes.length > 0) {
                proportionalPartsCount = countProportionsParts(rowDefinitions, proportionalIndexes);

                proportionalPartSize = notFixedHeight / proportionalPartsCount;
                calcProportionalItems(actualRowDefinitions, proportionalIndexes, proportionalPartSize);
            }

            return actualRowDefinitions;
        },

        getActualColWidths: function () {
            var colDefinitions = this.mvc.model.getColumnDefinition();
            var clientRect = this.calcClientRect();
            var fixedWidth = 0,
                notFixedWidth = 0,
                proportionalPartsCount = 0,
                proportionalPartSize = 0;
            var actualColDefinitions = colDefinitions.slice(0);

            //1. look for fixed size columns
            //2. look for auto size columns
            //3. look for proportional size columns
            var fixedIndexes = [];//getFixedSizeItems(colDefinitions);
            var autoIndexes = [];//getAutoSizeItems(colDefinitions);
            var proportionalIndexes = [];//getProportionalItems(colDefinitions);

            filterGridItems(colDefinitions, fixedIndexes, autoIndexes, proportionalIndexes)

            if (fixedIndexes.length > 0) {
                fixedWidth = sumItems(colDefinitions, fixedIndexes);
            }

            if (autoIndexes.length > 0) {
                fixedWidth += this.calcAutoColumnsWidths(actualColDefinitions, autoIndexes);
            }

            notFixedWidth = clientRect.width - fixedWidth;
            if (proportionalIndexes.length > 0) {
                proportionalPartsCount = countProportionsParts(colDefinitions, proportionalIndexes);

                proportionalPartSize = notFixedWidth / proportionalPartsCount;
                calcProportionalItems(actualColDefinitions, proportionalIndexes, proportionalPartSize);
            }

            return actualColDefinitions;
        },

        calcAutoColumnsWidths: function (colDefinitions, autoWidthCols) {
            var i, j;
            var result = 0;
            var colCells;
            var colSpan;
            var maxWidth;

            if (this.colCells.length === 0) {
                //not initialized
                return [];
            }

            for (i = 0; i < autoWidthCols.length; i++) {

                colCells = this.colCells[autoWidthCols[i]];
                maxWidth = 0;

                for (j = 0; j < colCells.length; j++) {
                    colSpan = colCells[j].control.model.get(Enums.ParameterRoles.COLSPAN) || 1;
                    if (colSpan <= 1) {
                        maxWidth = Math.max(maxWidth, colCells[j].control.controller.getControlWidthPx());
                    }
                }
                colDefinitions[autoWidthCols[i]] = maxWidth;
                result += maxWidth;
            }

            return result
        },

        calcAutoRowsHeights: function (rowDefinitions, autoHeightRows) {
            var i, j;
            var result = 0;
            var rowCells;
            var rowSpan;
            var maxHeight;

            if (this.rowCells.length === 0) {
                //not initialized
                return [];
            }

            for (i = 0; i < autoHeightRows.length; i++) {

                rowCells = this.rowCells[autoHeightRows[i]];
                maxHeight = 0;

                for (j = 0; j < rowCells.length; j++) {
                    rowSpan = rowCells[j].control.model.get(Enums.ParameterRoles.ROWSPAN) || 1;
                    if (rowSpan <= 1) {
                        maxHeight = Math.max(maxHeight, rowCells[j].control.controller.getControlHeightPx());
                    }
                }
                rowDefinitions[autoHeightRows[i]] = maxHeight;
                result += maxHeight;
            }

            return result
        },

        calcClientRect: function (startIndex) {
            var borderWidth = this.mvc.model.getBorderThickness(),
                left = 0,
                top = 0,
                width = this.getControlWidthPx() - borderWidth * 2,
                height = this.getControlHeightPx() - borderWidth * 2;

            return { left: left, top: top, width: width, height: height };
        }
    });

    function calcProportionalItems(definitions, proportionalItems, partSize) {
        var i,
            width,
            partsCount;

        for (i = 0; i < proportionalItems.length; i++) {
            width = definitions[proportionalItems[i]];
            if (width === '*') {
                partsCount = 1;
            }
            else {
                partsCount = parseInt(width);
            }

            definitions[proportionalItems[i]] = partsCount * partSize;
        }
    }

    function countProportionsParts(items, proportionalCols) {
        var i,
            result = 0,
            width;

        for (i = 0; i < proportionalCols.length; i++) {
            width = items[proportionalCols[i]];

            if (width === '*') {
                result++;
            }
            else {
                result += parseInt(width, 10);
            }
        }

        return result;
    }

    function sumItems(items, indexes) {
        var i;
        var sum = 0;

        for (i = 0; i < indexes.length; i++) {
            sum += parseFloat(items[indexes[i]]);
        }

        return sum;
    }

    function filterGridItems(items, fixedIndexes, autoIndexes, proportionalIndexes) {
        var i;
        for (i = 0; i < items.length; i++) {
            if (isFinite(items[i])) {
                fixedIndexes.push(i);
            }
            else if (items[i] === 'auto') {
                autoIndexes.push(i);
            }
            else {
                proportionalIndexes.push(i);
            }
        }
    }

    function getCellSideSize(cellIndex, cellSpan, cols) {
        if (cellSpan <= 1) {
            return cols[cellIndex];
        }

        var i,
            size = 0,
            cellsCount = cellIndex + cellSpan < cols.length ? cellSpan : cols.length - (cellIndex + 1);

        for (i = cellIndex; i <= cellIndex + cellsCount; i++) {
            size += cols[i];
        }

        return size;
    }

    function initEmptyItems(array, count) {
        var i;

        for (i = 0; i < count; i++) {
            if (array.length < i + 1) {
                array.push([]);
            }
            else if (!array[i]) {
                array[i] = [];
            }
        }
    }

    return GridPanelController;
});
